/**
 * Mosallas Admin JavaScript
 */

(function($) {
    'use strict';

    $(document).ready(function() {
        
        const $statusContainer = $('.mosallas-status');
        const $statusMeta = $statusContainer.find('.mosallas-status-meta');
        const defaultStatusMessage = $statusMeta.data('default-message') || '';

        const $connectionButton = $('#mosallas-test-connection');
        const defaultButtonText = $connectionButton.data('default-text') || $connectionButton.text() || 'تست اتصال';
        const loadingButtonText = $connectionButton.data('loading-text') || 'در حال بررسی...';
        $connectionButton.text(defaultButtonText);

        // Toggle API key visibility
        $('#mosallas-toggle-api-key').on('click', function() {
            const input = $('#mosallas_api_key');
            const type = input.attr('type');
            
            if (type === 'password') {
                input.attr('type', 'text');
                $(this).text('🔒');
            } else {
                input.attr('type', 'password');
                $(this).text('👁️');
            }
        });
        
        // Copy API key to clipboard
        $('#mosallas-copy-api-key').on('click', function() {
            const input = $('#mosallas_api_key');
            const apiKey = input.val();
            
            if (!apiKey) {
                showNotification('هیچ کلید API‌ای برای کپی کردن وجود ندارد', 'error');
                return;
            }
            
            // Copy to clipboard
            if (navigator.clipboard) {
                navigator.clipboard.writeText(apiKey).then(function() {
                    showNotification('کلید API کپی شد', 'success');
                }).catch(function() {
                    fallbackCopy(input[0]);
                });
            } else {
                fallbackCopy(input[0]);
            }
        });
        
        // Fallback copy method for older browsers
        function fallbackCopy(element) {
            const originalType = element.type;
            element.type = 'text';
            element.select();
            
            try {
                document.execCommand('copy');
                showNotification('کلید API کپی شد', 'success');
            } catch (err) {
                showNotification('خطا در کپی کردن', 'error');
            }
            
            element.type = originalType;
        }
        
        function requestConnectionConfirmation(options = {}) {
            const settings = Object.assign({
                silent: false,
                button: null,
            }, options);

            if (!mosallasAdmin.apiKey) {
                const message = 'لطفا ابتدا کلید API را ذخیره کنید';
                if (!settings.silent) {
                    showNotification(message, 'error');
                }
                return;
            }

            if (settings.button) {
                settings.button.prop('disabled', true).text(loadingButtonText);
            }

            console.log('� Verifying Mosallas connection via WordPress AJAX...');

            $.ajax({
                url: mosallasAdmin.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'mosallas_confirm_connection',
                    nonce: mosallasAdmin.nonce
                },
                success: function(response) {
                    if (settings.button) {
                        settings.button.prop('disabled', false).text(defaultButtonText);
                    }

                    if (response && response.success) {
                        const successMessage = (response.data && response.data.message)
                            ? response.data.message
                            : 'اتصال با موفقیت برقرار شد (' + new Date().toLocaleTimeString('fa-IR') + ')';

                        console.log('🎉 Connection confirmed by server.');
                        if (response.data && response.data.siteId) {
                            mosallasAdmin.siteId = response.data.siteId;
                        }
                        if (response.data && response.data.projectId) {
                            mosallasAdmin.projectId = response.data.projectId;
                        }
                        if (!settings.silent) {
                            showNotification('اتصال با موفقیت برقرار شد!', 'success');
                        }
                        updateConnectionStatus('connected', true, successMessage);
                    } else {
                        const warningMessage = (response && response.data && response.data.message)
                            ? response.data.message
                            : 'خطا در برقراری اتصال. لطفا تنظیمات را بررسی کنید.';

                        console.warn('⚠️ Connection confirmation failed:', response);
                        if (!settings.silent) {
                            showNotification(warningMessage, 'error');
                        }
                        updateConnectionStatus('error', true, warningMessage);
                    }
                },
                error: function(xhr) {
                    if (settings.button) {
                        settings.button.prop('disabled', false).text(defaultButtonText);
                    }

                    let errorMsg = 'خطا در تست اتصال';

                    if (xhr && xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                        errorMsg = xhr.responseJSON.data.message;
                    } else if (xhr && xhr.responseJSON && xhr.responseJSON.message) {
                        errorMsg = xhr.responseJSON.message;
                    } else if (xhr && xhr.responseText) {
                        errorMsg = xhr.responseText;
                    }

                    console.error('❌ Connection confirmation error:', errorMsg, xhr);

                    if (!settings.silent) {
                        showNotification(errorMsg, 'error');
                    }
                    updateConnectionStatus('error', true, errorMsg);
                }
            });
        }

        // Test connection button
        $('#mosallas-test-connection').on('click', function() {
            const apiKey = $('#mosallas_api_key').val();

            if (!apiKey) {
                console.error('❌ No API key provided');
                showNotification('لطفا ابتدا کلید API را وارد کنید', 'error');
                return;
            }

            if (apiKey !== mosallasAdmin.apiKey) {
                showNotification('لطفا ابتدا کلید API را ذخیره کرده و صفحه را دوباره بارگذاری کنید.', 'error');
                return;
            }

            requestConnectionConfirmation({ button: $(this) });
        });
        
        // Update connection status display
        function updateConnectionStatus(status, persist = true, message = '') {
            const statusEl = $('.mosallas-status');
            const statusIcon = statusEl.find('.mosallas-status-icon');
            const statusText = statusEl.find('.mosallas-status-text');
            const statusMeta = statusEl.find('.mosallas-status-meta');
            const fallbackMessage = statusMeta.data('default-message') || defaultStatusMessage;

            mosallasAdmin.connectionStatus = status;
            if (message) {
                mosallasAdmin.connectionMessage = message;
            }
            
            // Remove all status classes
            statusEl.removeClass('mosallas-status-connected mosallas-status-error mosallas-status-pending');
            
            // Add new status class
            statusEl.addClass('mosallas-status-' + status);
            
            // Update icon and text
            switch (status) {
                case 'connected':
                    statusIcon.text('✓');
                    statusText.text('متصل به Mosallas.ir');
                    break;
                case 'error':
                    statusIcon.text('✗');
                    statusText.text('خطا در اتصال');
                    break;
                case 'pending':
                    statusIcon.text('⏳');
                    statusText.text('در انتظار اتصال');
                    break;
            }
            
            if (message) {
                statusMeta.text(message);
                statusMeta.data('default-message', message);
            } else {
                statusMeta.text(fallbackMessage);
            }
            
            // Send status update to WordPress options (via AJAX)
            if (persist) {
                $.post(mosallasAdmin.ajaxUrl, {
                    action: 'mosallas_update_connection_status',
                    nonce: mosallasAdmin.nonce,
                    status: status,
                    message: message
                });
            }
        }
        
        // Show notification
        function showNotification(message, type) {
            // Remove existing notifications
            $('.mosallas-notification').remove();
            
            // Create notification element
            const notification = $('<div>')
                .addClass('mosallas-notification')
                .addClass(type)
                .text(message);
            
            // Add to body
            $('body').append(notification);
            
            // Show with animation
            setTimeout(function() {
                notification.addClass('show');
            }, 10);
            
            // Auto hide after 3 seconds
            setTimeout(function() {
                notification.removeClass('show');
                setTimeout(function() {
                    notification.remove();
                }, 300);
            }, 3000);
        }
        
        // Check connection status on page load
        if (mosallasAdmin.apiKey) {
            console.log('🔄 Auto-checking connection on page load...');
            console.log('📝 Stored API Key exists:', !!mosallasAdmin.apiKey);
            updateConnectionStatus(mosallasAdmin.connectionStatus || 'pending', false, mosallasAdmin.connectionMessage || '');

            const isProvisioned = !!(mosallasAdmin.siteId && mosallasAdmin.projectId);

            if ((mosallasAdmin.connectionStatus || 'pending') !== 'connected' || !isProvisioned) {
                requestConnectionConfirmation({ silent: true });
            }
        } else {
            console.log('⏸️ Skipping auto-check (missing API key or webhook URL)');
            console.log('Has API Key:', !!mosallasAdmin.apiKey);
            updateConnectionStatus('pending', false, 'برای فعال‌سازی، ابتدا تنظیمات را ذخیره کنید.');
        }
        
        console.log('🎨 Mosallas Admin JavaScript loaded successfully');
        console.log('📊 Available data:', {
            ajaxUrl: mosallasAdmin.ajaxUrl,
            restUrl: mosallasAdmin.restUrl,
            hasApiKey: !!mosallasAdmin.apiKey,
            webhookUrl: mosallasAdmin.webhookUrl,
            siteUrl: mosallasAdmin.siteUrl,
            siteName: mosallasAdmin.siteName
        });
    });

    // =====================================
    // Logs Management
    // =====================================
    
    let autoRefreshInterval = null;
    
    /**
     * Load and display logs
     */
    function loadLogs() {
        const level = $('#mosallas-log-level').val();
        const search = $('#mosallas-log-search').val();
        
        const $container = $('#mosallas-logs-container');
        $container.html('<div class="mosallas-logs-loading"><span class="spinner is-active"></span><p>در حال بارگذاری لاگ‌ها...</p></div>');
        
        $.ajax({
            url: mosallasAdmin.ajaxUrl,
            method: 'POST',
            data: {
                action: 'mosallas_get_logs',
                nonce: mosallasAdmin.nonce,
                level: level,
                search: search,
                limit: 100
            },
            success: function(response) {
                if (response.success && response.data.logs) {
                    displayLogs(response.data.logs);
                    $('#mosallas-logs-count').text(response.data.count + ' لاگ');
                    $('#mosallas-logs-last-update').text(new Date().toLocaleTimeString('fa-IR'));
                } else {
                    $container.html('<div class="mosallas-logs-empty"><div class="mosallas-logs-empty-icon">📋</div><p>هیچ لاگی یافت نشد</p></div>');
                }
            },
            error: function() {
                $container.html('<div class="mosallas-logs-empty"><div class="mosallas-logs-empty-icon">❌</div><p>خطا در بارگذاری لاگ‌ها</p></div>');
            }
        });
    }
    
    /**
     * Display logs in the container
     */
    function displayLogs(logs) {
        const $container = $('#mosallas-logs-container');
        
        if (!logs || logs.length === 0) {
            $container.html('<div class="mosallas-logs-empty"><div class="mosallas-logs-empty-icon">📋</div><p>هیچ لاگی یافت نشد</p></div>');
            return;
        }
        
        let html = '';
        logs.forEach(function(log) {
            const levelClass = 'level-' + log.level;
            const logClass = 'log-' + log.level;
            const levelText = {
                'info': 'اطلاعات',
                'success': 'موفقیت',
                'warning': 'هشدار',
                'error': 'خطا'
            }[log.level] || log.level;
            
            html += '<div class="mosallas-log-entry ' + logClass + '">';
            html += '<div class="mosallas-log-header">';
            html += '<div class="mosallas-log-message">' + escapeHtml(log.message) + '</div>';
            html += '<span class="mosallas-log-level ' + levelClass + '">' + levelText + '</span>';
            html += '</div>';
            html += '<div class="mosallas-log-time">' + log.timestamp + '</div>';
            
            if (log.context && Object.keys(log.context).length > 0) {
                html += '<div class="mosallas-log-context">';
                html += '<pre>' + JSON.stringify(log.context, null, 2) + '</pre>';
                html += '</div>';
            }
            
            html += '</div>';
        });
        
        $container.html(html);
    }
    
    /**
     * Escape HTML to prevent XSS
     */
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
    // Refresh logs button
    $('#mosallas-refresh-logs').on('click', function() {
        loadLogs();
    });
    
    // Clear logs button
    $('#mosallas-clear-logs').on('click', function() {
        if (!confirm('آیا مطمئن هستید که می‌خواهید تمام لاگ‌ها را پاک کنید؟')) {
            return;
        }
        
        $.ajax({
            url: mosallasAdmin.ajaxUrl,
            method: 'POST',
            data: {
                action: 'mosallas_clear_logs',
                nonce: mosallasAdmin.nonce
            },
            success: function(response) {
                if (response.success) {
                    alert('تمام لاگ‌ها پاک شدند');
                    loadLogs();
                }
            },
            error: function() {
                alert('خطا در پاک کردن لاگ‌ها');
            }
        });
    });
    
    // Download logs button
    $('#mosallas-download-logs').on('click', function() {
        window.location.href = mosallasAdmin.ajaxUrl + '?action=mosallas_download_logs&nonce=' + mosallasAdmin.nonce;
    });
    
    // Filter logs by level
    $('#mosallas-log-level').on('change', function() {
        loadLogs();
    });
    
    // Search logs
    let searchTimeout;
    $('#mosallas-log-search').on('input', function() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(function() {
            loadLogs();
        }, 500);
    });
    
    // Auto-refresh logs
    $('#mosallas-auto-refresh-logs').on('change', function() {
        if ($(this).is(':checked')) {
            autoRefreshInterval = setInterval(function() {
                loadLogs();
            }, 5000);
        } else {
            if (autoRefreshInterval) {
                clearInterval(autoRefreshInterval);
                autoRefreshInterval = null;
            }
        }
    });
    
    // Load logs on page load
    if ($('#mosallas-logs-container').length > 0) {
        loadLogs();
    }

})(jQuery);
