<?php
/**
 * Mosallas Admin Interface
 * Handles WordPress admin menu and settings page
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mosallas_Admin {
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            'Mosallas.ir',              // Page title
            'Mosallas',                 // Menu title
            'manage_options',           // Capability
            'mosallas-settings',        // Menu slug
            array($this, 'settings_page'), // Callback
            'dashicons-edit',           // Icon
            30                          // Position
        );
    }
    
    /**
     * Render settings page
     */
    public function settings_page() {
        // Handle form submission
        if (isset($_POST['mosallas_save_settings']) && 
            check_admin_referer('mosallas_settings_action', 'mosallas_settings_nonce')) {
            
            $api_key = sanitize_text_field($_POST['mosallas_api_key'] ?? '');
            $webhook_url = esc_url_raw($_POST['mosallas_webhook_url'] ?? '');
            
            update_option('mosallas_api_key', $api_key);
            update_option('mosallas_webhook_url', $webhook_url);
            update_option('mosallas_connection_status', 'pending');
            update_option('mosallas_connection_message', __('برای فعال‌سازی ارتباط، روی دکمه «اتصال به وبسایت» کلیک کنید.', 'mosallas'));
            update_option('mosallas_connection_error', '');
            update_option('mosallas_connection_updated_at', current_time('mysql'));
            
            echo '<div class="notice notice-success is-dismissible"><p>تنظیمات با موفقیت ذخیره شد.</p></div>';
        }
        
        // Include settings page template
        include MOSALLAS_PLUGIN_DIR . 'admin/settings-page.php';
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our plugin pages
        if ($hook !== 'toplevel_page_mosallas-settings') {
            return;
        }
        wp_enqueue_style(
            'mosallas-admin',
            MOSALLAS_PLUGIN_URL . 'admin/assets/css/admin.css',
            array(),
            MOSALLAS_VERSION
        );

        wp_enqueue_script(
            'mosallas-admin',
            MOSALLAS_PLUGIN_URL . 'admin/assets/js/admin.js',
            array('jquery'),
            MOSALLAS_VERSION,
            true
        );

        $api_key = get_option('mosallas_api_key', '');
        $webhook_url = get_option('mosallas_webhook_url', 'https://mosallas.ir/api/wordpress/webhook');
        $connection_status = get_option('mosallas_connection_status', 'pending');
        $connection_message = get_option('mosallas_connection_message', '');
        $site_id = get_option('mosallas_site_id', '');
        $project_id = get_option('mosallas_project_id', '');

        wp_localize_script(
            'mosallas-admin',
            'mosallasAdmin',
            array(
                'ajaxUrl'            => admin_url('admin-ajax.php'),
                'restUrl'            => esc_url_raw(rest_url('mosallas/v1')),
                'nonce'              => wp_create_nonce('mosallas_admin_nonce'),
                'apiKey'             => $api_key,
                'webhookUrl'         => $webhook_url,
                'siteUrl'            => get_site_url(),
                'siteName'           => get_bloginfo('name'),
                'connectionStatus'   => $connection_status,
                'connectionMessage'  => $connection_message,
                'siteId'             => $site_id,
                'projectId'          => $project_id,
                'connectButtonLabel' => __('اتصال به وبسایت', 'mosallas'),
            )
        );
    }
    
    /**
     * AJAX handler to get logs
     */
    public function ajax_get_logs() {
        check_ajax_referer('mosallas_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Unauthorized'), 403);
        }
        
        $level = sanitize_text_field($_POST['level'] ?? 'all');
        $search = sanitize_text_field($_POST['search'] ?? '');
        $limit = intval($_POST['limit'] ?? 100);
        
        $logs = Mosallas_Logger::get_logs($level, $search, $limit);
        $count = Mosallas_Logger::get_logs_count($level);
        
        wp_send_json_success(array(
            'logs' => $logs,
            'count' => $count,
            'timestamp' => current_time('mysql')
        ));
    }
    
    /**
     * AJAX handler to clear logs
     */
    public function ajax_clear_logs() {
        check_ajax_referer('mosallas_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Unauthorized'), 403);
        }
        
        Mosallas_Logger::clear_logs();
        
        wp_send_json_success(array(
            'message' => 'تمام لاگ‌ها پاک شدند'
        ));
    }

    /**
     * AJAX handler to update connection status
     */
    public function ajax_update_connection_status() {
        check_ajax_referer('mosallas_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Unauthorized'), 403);
        }

        $status = sanitize_text_field($_POST['status'] ?? 'pending');
        $message = isset($_POST['message']) ? sanitize_textarea_field(wp_unslash($_POST['message'])) : '';

        $allowed_statuses = array('connected', 'error', 'pending');
        if (!in_array($status, $allowed_statuses, true)) {
            $status = 'pending';
        }

        update_option('mosallas_connection_status', $status);

        if ($message !== '') {
            update_option('mosallas_connection_message', $message);
        } else {
            delete_option('mosallas_connection_message');
        }

        update_option('mosallas_connection_updated_at', current_time('mysql'));

        if ('error' === $status) {
            update_option('mosallas_connection_error', $message);
        } elseif ('connected' === $status) {
            update_option('mosallas_connection_error', '');
        }

        wp_send_json_success(array(
            'status'  => $status,
            'message' => $message,
        ));
    }

    /**
     * AJAX handler to confirm connection via server-side request
     */
    public function ajax_confirm_connection() {
        check_ajax_referer('mosallas_admin_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Unauthorized'), 403);
        }

        $api = new Mosallas_API();
        $result = $api->confirm_connection();

        if (!is_array($result)) {
            $result = array(
                'success' => (bool) $result,
            );
        }

        if (!empty($result['success'])) {
            wp_send_json_success(array(
                'message' => $result['message'] ?? __('اتصال با موفقیت تایید شد.', 'mosallas'),
                'siteId' => $result['siteId'] ?? get_option('mosallas_site_id'),
                'projectId' => $result['projectId'] ?? get_option('mosallas_project_id'),
                'response' => $result['response'] ?? null,
            ));
        }

        $status_code = !empty($result['http_status']) ? intval($result['http_status']) : 500;

        wp_send_json_error(array(
            'message' => $result['message'] ?? __('تایید اتصال ناموفق بود.', 'mosallas'),
            'response' => $result['response'] ?? null,
        ), $status_code);
    }
    
    /**
     * AJAX handler to download logs
     */
    public function ajax_download_logs() {
        check_ajax_referer('mosallas_admin_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $logs_json = Mosallas_Logger::export_logs();
        
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="mosallas-logs-' . date('Y-m-d-His') . '.json"');
        header('Content-Length: ' . strlen($logs_json));
        
        echo $logs_json;
        exit;
    }
}
