<?php
/**
 * Mosallas API Handler
 * Handles REST API endpoints for communication with Mosallas.ir
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mosallas_API {
    
    /**
     * Mosallas.ir API base URL
     */
    private $api_base = 'https://mosallas.ir/api/wordpress';
    
    /**
     * Register REST API routes
     */
    public function register_routes() {
        // Endpoint for Mosallas.ir to publish articles
        register_rest_route('mosallas/v1', '/publish', array(
            'methods' => 'POST',
            'callback' => array($this, 'publish_article'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
        
        // Endpoint to get existing posts for internal linking
        register_rest_route('mosallas/v1', '/posts', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_existing_posts'),
            'permission_callback' => array($this, 'verify_api_key')
        ));

        // Endpoint to update existing posts on demand
        register_rest_route('mosallas/v1', '/posts/(?P<id>\\d+)', array(
            'methods' => array('PUT', 'PATCH'),
            'callback' => array($this, 'update_post'),
            'permission_callback' => array($this, 'verify_api_key')
        ));

        // Endpoint to delete existing posts on demand
        register_rest_route('mosallas/v1', '/posts/(?P<id>\\d+)', array(
            'methods' => 'DELETE',
            'callback' => array($this, 'delete_post'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
        
        // Endpoint to get articles for internal linking from database
        register_rest_route('mosallas/v1', '/internal-links/articles', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_internal_link_articles'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
        
        // Endpoint to get products for internal linking from database
        register_rest_route('mosallas/v1', '/internal-links/products', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_internal_link_products'),
            'permission_callback' => array($this, 'verify_api_key')
        ));

        // Endpoint to fetch plugin version and site status
        register_rest_route('mosallas/v1', '/version', array(
            'methods' => 'GET',
            'callback' => array($this, 'get_plugin_version'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
        
        // Endpoint to sync articles to database
        register_rest_route('mosallas/v1', '/sync/articles', array(
            'methods' => 'POST',
            'callback' => array($this, 'sync_articles_to_db'),
            'permission_callback' => array($this, 'verify_api_key')
        ));
        
        // Endpoint to sync products to database
        register_rest_route('mosallas/v1', '/sync/products', array(
            'methods' => 'POST',
            'callback' => array($this, 'sync_products_to_db'),
            'permission_callback' => array($this, 'verify_api_key')
        ));

        // Endpoints to manage categories remotely
        register_rest_route('mosallas/v1', '/categories', array(
            array(
                'methods' => 'GET',
                'callback' => array($this, 'get_categories'),
                'permission_callback' => array($this, 'verify_api_key')
            ),
            array(
                'methods' => 'POST',
                'callback' => array($this, 'create_category'),
                'permission_callback' => array($this, 'verify_api_key')
            )
        ));

        register_rest_route('mosallas/v1', '/categories/(?P<id>\d+)', array(
            array(
                'methods' => array('PUT', 'PATCH'),
                'callback' => array($this, 'update_category'),
                'permission_callback' => array($this, 'verify_api_key')
            ),
            array(
                'methods' => 'DELETE',
                'callback' => array($this, 'delete_category'),
                'permission_callback' => array($this, 'verify_api_key')
            )
        ));
    }

    /**
     * Return plugin version and basic site status
     */
    public function get_plugin_version($request) {
        return new WP_REST_Response(array(
            'success' => true,
            'pluginVersion' => MOSALLAS_VERSION,
            'phpVersion' => PHP_VERSION,
            'wpVersion' => get_bloginfo('version'),
            'siteUrl' => get_site_url(),
            'siteId' => get_option('mosallas_site_id'),
            'connectionStatus' => get_option('mosallas_connection_status'),
            'connectionUpdatedAt' => get_option('mosallas_connection_updated_at'),
        ), 200);
    }
    
    /**
     * Verify API key from request header
     */
    public function verify_api_key($request) {
        $api_key = $request->get_header('X-API-Key');
        $stored_key = get_option('mosallas_api_key');
        
        if (empty($stored_key)) {
            return new WP_Error(
                'no_api_key',
                __('API Key not configured', 'mosallas'),
                array('status' => 401)
            );
        }
        
        if ($api_key !== $stored_key) {
            return new WP_Error(
                'invalid_api_key',
                __('Invalid API Key', 'mosallas'),
                array('status' => 401)
            );
        }
        
        return true;
    }
    
    /**
     * Publish article from Mosallas.ir
     */
    public function publish_article($request) {
        try {
            $data = $request->get_json_params();
            
            // Log raw incoming data for debugging
            error_log('Mosallas API: Received publish request');
            error_log('Mosallas API: Raw data - ' . print_r($data, true));
            
            // Log incoming request
            Mosallas_Logger::info('📥 درخواست انتشار مقاله دریافت شد', array(
                'article_id' => $data['articleId'] ?? 'unknown',
                'title' => $data['title'] ?? 'unknown',
                'has_content' => !empty($data['content']),
                'content_length' => !empty($data['content']) ? strlen($data['content']) : 0
            ));
            
            // Validate required fields
            if (empty($data['title'])) {
                error_log('Mosallas API: Missing title field');
                Mosallas_Logger::error('❌ فیلد عنوان ارسال نشده است');
                return new WP_REST_Response(array(
                    'success' => false,
                    'error' => __('Missing required field: title', 'mosallas')
                ), 400);
            }
            
            if (empty($data['content'])) {
                error_log('Mosallas API: Missing content field');
                Mosallas_Logger::error('❌ فیلد محتوا ارسال نشده است');
                return new WP_REST_Response(array(
                    'success' => false,
                    'error' => __('Missing required field: content', 'mosallas')
                ), 400);
            }
            
            error_log('Mosallas API: Validation passed, creating publisher instance');
            
            $publisher = new Mosallas_Publisher();
            
            error_log('Mosallas API: Calling create_post');
            $post_id = $publisher->create_post($data);
            
            error_log('Mosallas API: Post created successfully with ID: ' . $post_id);
            
            Mosallas_Logger::success('✅ مقاله با موفقیت منتشر شد', array(
                'post_id' => $post_id,
                'title' => $data['title'],
                'article_id' => $data['articleId'] ?? 'unknown'
            ));
            
            // Notify Mosallas.ir of successful publish
            $this->notify_publish_success($data['articleId'] ?? '', $post_id);
            
            // Collect featured image details
            $thumb_id = get_post_thumbnail_id($post_id);
            $thumb_url = '';
            if ($thumb_id) {
                $src = wp_get_attachment_image_src($thumb_id, 'full');
                if ($src && is_array($src)) {
                    $thumb_url = $src[0];
                }
            }
            
            return new WP_REST_Response(array(
                'success' => true,
                'post_id' => $post_id,
                'post_url' => get_permalink($post_id),
                'edit_url' => get_edit_post_link($post_id, 'raw'),
                'featuredImageId' => $thumb_id,
                'featuredImageUrl' => $thumb_url
            ), 200);
            
        } catch (Exception $e) {
            error_log('Mosallas API: Exception caught - ' . $e->getMessage());
            error_log('Mosallas API: Exception trace - ' . $e->getTraceAsString());
            
            Mosallas_Logger::error('❌ خطا در انتشار مقاله: ' . $e->getMessage(), array(
                'article_id' => isset($data) && isset($data['articleId']) ? $data['articleId'] : 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
            
            return new WP_REST_Response(array(
                'success' => false,
                'error' => $e->getMessage(),
                'details' => array(
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                )
            ), 500);
        } catch (Throwable $t) {
            // Catch fatal errors as well
            error_log('Mosallas API: Fatal error - ' . $t->getMessage());
            error_log('Mosallas API: Fatal error trace - ' . $t->getTraceAsString());
            
            Mosallas_Logger::error('❌ خطای جدی در انتشار مقاله: ' . $t->getMessage(), array(
                'error' => $t->getMessage(),
                'trace' => $t->getTraceAsString()
            ));
            
            return new WP_REST_Response(array(
                'success' => false,
                'error' => 'Internal server error: ' . $t->getMessage(),
                'details' => array(
                    'file' => $t->getFile(),
                    'line' => $t->getLine()
                )
            ), 500);
        }
    }
    
    /**
     * Get existing posts for internal linking
     */
    public function get_existing_posts($request) {
        $args = array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => 100,
            'orderby' => 'date',
            'order' => 'DESC'
        );
        
        $posts = get_posts($args);
        $result = array();
        
        foreach ($posts as $post) {
            $result[] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'url' => get_permalink($post->ID),
                'focusKeyword' => $this->get_focus_keyword($post->ID)
            );
        }
        
        return new WP_REST_Response(array(
            'success' => true,
            'posts' => $result
        ), 200);
    }

    /**
     * Update an existing post with new content received from Mosallas.ir
     */
    public function update_post($request) {
        $post_id = intval($request->get_param('id'));

        if ($post_id <= 0) {
            return new WP_Error(
                'invalid_post_id',
                __('Invalid post ID provided', 'mosallas'),
                array('status' => 400)
            );
        }

        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error(
                'post_not_found',
                __('Post not found', 'mosallas'),
                array('status' => 404)
            );
        }

        $data = $request->get_json_params();
        if (!is_array($data)) {
            $data = array();
        }

        try {
            $publisher = new Mosallas_Publisher();
            $result = $publisher->update_existing_post($post_id, $data);

            Mosallas_Logger::success('✏️ مقاله با درخواست Mosallas بروزرسانی شد', array(
                'post_id' => $post_id,
                'article_id' => $data['articleId'] ?? 'unknown'
            ));

            return new WP_REST_Response(array(
                'success' => true,
                'post_id' => $result['post_id'],
                'post_url' => $result['post_url'],
                'edit_url' => $result['edit_url'],
                'featuredImageId' => $result['featuredImageId'],
                'featuredImageUrl' => $result['featuredImageUrl']
            ), 200);
        } catch (Exception $e) {
            Mosallas_Logger::error('❌ خطا در بروزرسانی مقاله', array(
                'post_id' => $post_id,
                'error' => $e->getMessage()
            ));

            return new WP_REST_Response(array(
                'success' => false,
                'error' => $e->getMessage()
            ), 500);
        }
    }

    /**
     * Delete a post from WordPress when Mosallas requests it
     */
    public function delete_post($request) {
        $post_id = intval($request->get_param('id'));

        if ($post_id <= 0) {
            return new WP_Error(
                'invalid_post_id',
                __('Invalid post ID provided', 'mosallas'),
                array('status' => 400)
            );
        }

        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error(
                'post_not_found',
                __('Post not found', 'mosallas'),
                array('status' => 404)
            );
        }

        $force_param = $request->get_param('force');
        $force_delete = true;

        if (null !== $force_param) {
            $force_delete = filter_var($force_param, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
            if ($force_delete === null) {
                $force_delete = true;
            }
        }

        $deleted = wp_delete_post($post_id, $force_delete);

        if (!$deleted) {
            Mosallas_Logger::error('❌ حذف مقاله ناموفق بود', array(
                'post_id' => $post_id
            ));

            return new WP_REST_Response(array(
                'success' => false,
                'error' => __('Failed to delete post', 'mosallas')
            ), 500);
        }

        Mosallas_Logger::warning('🗑️ مقاله با درخواست Mosallas حذف شد', array(
            'post_id' => $post_id,
            'force' => $force_delete
        ));

        return new WP_REST_Response(array(
            'success' => true,
            'post_id' => $post_id,
            'force' => $force_delete
        ), 200);
    }
    
    /**
     * Get focus keyword from SEO plugins
     */
    private function get_focus_keyword($post_id) {
        // Try Yoast SEO
        $yoast_keyword = get_post_meta($post_id, '_yoast_wpseo_focuskw', true);
        if (!empty($yoast_keyword)) {
            return $yoast_keyword;
        }
        
        // Try Rank Math
        $rankmath_keyword = get_post_meta($post_id, 'rank_math_focus_keyword', true);
        if (!empty($rankmath_keyword)) {
            return $rankmath_keyword;
        }
        
        // Fallback to post title
        return get_the_title($post_id);
    }
    
    /**
     * Confirm connection with Mosallas.ir on activation
     */
    public function confirm_connection() {
        $api_key = get_option('mosallas_api_key');

        if (empty($api_key)) {
            $message = __('کلید API تنظیم نشده است.', 'mosallas');
            Mosallas_Logger::warning('⚠️ ' . $message);

            return array(
                'success' => false,
                'message' => $message,
            );
        }

        Mosallas_Logger::info('🔗 در حال تایید اتصال با Mosallas.ir...', array(
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name')
        ));

        $payload = array(
            'action' => 'confirm_connection',
            'data' => array(
                'siteId' => get_option('mosallas_site_id'),
                'siteUrl' => get_site_url(),
                'siteName' => get_bloginfo('name')
            )
        );

        $response = wp_remote_post($this->api_base . '/webhook', array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => wp_json_encode($payload),
            'timeout' => 15
        ));

        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            Mosallas_Logger::error('❌ خطا در اتصال به Mosallas.ir', array(
                'error' => $error_message
            ));
            update_option('mosallas_connection_status', 'error');
            update_option('mosallas_connection_error', $error_message);
            update_option('mosallas_connection_message', $error_message);
            update_option('mosallas_connection_updated_at', current_time('mysql'));

            return array(
                'success' => false,
                'message' => $error_message,
                'error' => $error_message,
            );
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $http_status = wp_remote_retrieve_response_code($response);

        if (isset($body['success']) && $body['success']) {
            $site_id = !empty($body['siteId']) ? sanitize_text_field($body['siteId']) : get_option('mosallas_site_id');
            $project_id = !empty($body['projectId']) ? sanitize_text_field($body['projectId']) : get_option('mosallas_project_id');

            // Optional: allow backend to send project system prompt for llms.txt generation.
            $project_system_prompt = '';
            if (!empty($body['projectSystemPrompt'])) {
                $project_system_prompt = sanitize_textarea_field(wp_unslash($body['projectSystemPrompt']));
            } elseif (!empty($body['systemPrompt'])) {
                $project_system_prompt = sanitize_textarea_field(wp_unslash($body['systemPrompt']));
            }

            if (!empty($body['siteId'])) {
                update_option('mosallas_site_id', $site_id);
            }
            if (!empty($body['projectId'])) {
                update_option('mosallas_project_id', $project_id);
            }
            if (!empty($project_system_prompt)) {
                update_option('mosallas_project_system_prompt', $project_system_prompt);
            }

            $success_message = __('اتصال به Mosallas.ir با موفقیت تایید شد.', 'mosallas');
            Mosallas_Logger::success('✅ ' . $success_message);

            update_option('mosallas_connection_status', 'connected');
            update_option('mosallas_connection_error', '');
            update_option('mosallas_connection_message', $success_message);
            update_option('mosallas_connection_updated_at', current_time('mysql'));

            // Best-effort: refresh llms.txt via webhook and keep a real file in the site root.
            if (class_exists('Mosallas_LLMs_Txt')) {
                $refresh = Mosallas_LLMs_Txt::refresh_llms_txt_via_webhook();
                if (!empty($refresh['success'])) {
                    Mosallas_Logger::info('📄 llms.txt refreshed', $refresh);
                } else {
                    Mosallas_Logger::warning('⚠️ llms.txt refresh failed', $refresh);
                }

                $write = Mosallas_LLMs_Txt::write_llms_txt_to_root();
                if (!empty($write['success'])) {
                    Mosallas_Logger::info('📄 llms.txt written', $write);
                } else {
                    Mosallas_Logger::warning('⚠️ llms.txt write failed', $write);
                }
            }

            return array(
                'success' => true,
                'message' => $success_message,
                'siteId' => $site_id,
                'projectId' => $project_id,
                'response' => $body,
                'http_status' => $http_status,
            );
        }

        $failure_message = __('تایید اتصال با Mosallas.ir ناموفق بود.', 'mosallas');
        if (!empty($body['message'])) {
            $failure_message = sanitize_text_field($body['message']);
        }

        Mosallas_Logger::error('❌ ' . $failure_message, array(
            'response' => $body
        ));

        update_option('mosallas_connection_status', 'error');
        update_option('mosallas_connection_error', $failure_message);
        update_option('mosallas_connection_message', $failure_message);
        update_option('mosallas_connection_updated_at', current_time('mysql'));

        return array(
            'success' => false,
            'message' => $failure_message,
            'response' => $body,
            'http_status' => $http_status,
        );
    }
    
    /**
     * Notify Mosallas.ir of successful article publication
     */
    private function notify_publish_success($article_id, $post_id) {
        $api_key = get_option('mosallas_api_key');
        
        if (empty($api_key) || empty($article_id)) {
            return;
        }
        
        // Resolve featured image details
        $thumb_id = get_post_thumbnail_id($post_id);
        $thumb_url = '';
        if ($thumb_id) {
            $src = wp_get_attachment_image_src($thumb_id, 'full');
            if ($src && is_array($src)) {
                $thumb_url = $src[0];
            }
        }
        
        wp_remote_post($this->api_base . '/webhook', array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array(
                'action' => 'confirm_publish',
                'data' => array(
                    'articleId' => $article_id,
                    'wpPostId' => $post_id,
                    'wpPostUrl' => get_permalink($post_id),
                    'wpEditUrl' => get_edit_post_link($post_id, 'raw'),
                    'featuredImageId' => $thumb_id,
                    'featuredImageUrl' => $thumb_url
                )
            )),
            'timeout' => 15,
            'blocking' => false // Don't wait for response
        ));
    }
    
    /**
     * Get articles for internal linking from database
     */
    public function get_internal_link_articles($request) {
        $database = new Mosallas_Database();
        $query = $request->get_param('query') ?: '';
        $limit = intval($request->get_param('limit') ?: 10);
        
        $articles = $database->get_articles_for_linking($query, $limit);
        
        return new WP_REST_Response(array(
            'success' => true,
            'articles' => $articles
        ), 200);
    }
    
    /**
     * Get products for internal linking from database
     */
    public function get_internal_link_products($request) {
        $database = new Mosallas_Database();
        $query = $request->get_param('query') ?: '';
        $limit = intval($request->get_param('limit') ?: 10);
        
        $products = $database->get_products_for_linking($query, $limit);
        
        return new WP_REST_Response(array(
            'success' => true,
            'products' => $products
        ), 200);
    }
    
    /**
     * Sync articles to database
     */
    public function sync_articles_to_db($request) {
        $database = new Mosallas_Database();
        $synced_count = $database->sync_articles();
        
        return new WP_REST_Response(array(
            'success' => true,
            'synced_count' => $synced_count,
            'message' => "Successfully synced {$synced_count} articles to database"
        ), 200);
    }
    
    /**
     * Sync products to database
     */
    public function sync_products_to_db($request) {
        $database = new Mosallas_Database();
        $synced_count = $database->sync_products();
        
        return new WP_REST_Response(array(
            'success' => true,
            'synced_count' => $synced_count,
            'message' => "Successfully synced {$synced_count} products to database"
        ), 200);
    }

    /**
     * Get categories for a taxonomy (defaults to post categories)
     */
    public function get_categories($request) {
        $taxonomy = $this->resolve_taxonomy($request->get_param('taxonomy'));

        $args = array(
            'taxonomy' => $taxonomy,
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC'
        );

        $search = $request->get_param('search');
        if (!empty($search)) {
            $args['search'] = sanitize_text_field($search);
        }

        $parent = $request->get_param('parent');
        if ($parent !== null) {
            $args['parent'] = intval($parent);
        }

        $terms = get_terms($args);

        if (is_wp_error($terms)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => $terms->get_error_message()
            ), 400);
        }

        $categories = array_map(function($term) use ($taxonomy) {
            return $this->format_category($term, $taxonomy);
        }, $terms);

        return new WP_REST_Response(array(
            'success' => true,
            'taxonomy' => $taxonomy,
            'categories' => $categories
        ), 200);
    }

    /**
     * Create a new category
     */
    public function create_category($request) {
        $taxonomy = $this->resolve_taxonomy($request->get_param('taxonomy'));
        $data = $request->get_json_params();

        $name = isset($data['name']) ? sanitize_text_field($data['name']) : '';
        if (empty($name)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => __('Category name is required', 'mosallas')
            ), 400);
        }

        $args = array();

        if (!empty($data['slug'])) {
            $args['slug'] = sanitize_title($data['slug']);
        }

        if (isset($data['description'])) {
            $args['description'] = sanitize_text_field($data['description']);
        }

        if (!empty($data['parentId'])) {
            $parent_id = intval($data['parentId']);
            $parent_term = get_term($parent_id, $taxonomy);
            if (!$parent_term || is_wp_error($parent_term)) {
                return new WP_REST_Response(array(
                    'success' => false,
                    'error' => __('Parent category not found', 'mosallas')
                ), 404);
            }
            $args['parent'] = $parent_id;
        }

        $result = wp_insert_term($name, $taxonomy, $args);

        if (is_wp_error($result)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => $result->get_error_message()
            ), 400);
        }

        $term_id = $result['term_id'];
        $term = get_term($term_id, $taxonomy);

        return new WP_REST_Response(array(
            'success' => true,
            'category' => $this->format_category($term, $taxonomy)
        ), 201);
    }

    /**
     * Update an existing category
     */
    public function update_category($request) {
        $taxonomy = $this->resolve_taxonomy($request->get_param('taxonomy'));
        $term_id = intval($request->get_param('id'));
        $data = $request->get_json_params();

        if (!$term_id) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => __('Invalid category ID', 'mosallas')
            ), 400);
        }

        $existing_term = get_term($term_id, $taxonomy);
        if (!$existing_term || is_wp_error($existing_term)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => __('Category not found', 'mosallas')
            ), 404);
        }

        $args = array();

        if (isset($data['name'])) {
            $name = sanitize_text_field($data['name']);
            if (empty($name)) {
                return new WP_REST_Response(array(
                    'success' => false,
                    'error' => __('Category name cannot be empty', 'mosallas')
                ), 400);
            }
            $args['name'] = $name;
        }

        if (isset($data['slug'])) {
            $slug = sanitize_title($data['slug']);
            if (!empty($slug)) {
                $args['slug'] = $slug;
            }
        }

        if (array_key_exists('description', $data)) {
            $args['description'] = sanitize_text_field($data['description']);
        }

        if (array_key_exists('parentId', $data)) {
            $parent_id = intval($data['parentId']);

            if ($parent_id === $term_id) {
                return new WP_REST_Response(array(
                    'success' => false,
                    'error' => __('A category cannot be its own parent', 'mosallas')
                ), 400);
            }

            if ($parent_id > 0) {
                $parent_term = get_term($parent_id, $taxonomy);
                if (!$parent_term || is_wp_error($parent_term)) {
                    return new WP_REST_Response(array(
                        'success' => false,
                        'error' => __('Parent category not found', 'mosallas')
                    ), 404);
                }
                $args['parent'] = $parent_id;
            } else {
                $args['parent'] = 0;
            }
        }

        if (empty($args)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => __('No fields provided for update', 'mosallas')
            ), 400);
        }

        $result = wp_update_term($term_id, $taxonomy, $args);

        if (is_wp_error($result)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => $result->get_error_message()
            ), 400);
        }

        $term = get_term($term_id, $taxonomy);

        return new WP_REST_Response(array(
            'success' => true,
            'category' => $this->format_category($term, $taxonomy)
        ), 200);
    }

    /**
     * Delete a category
     */
    public function delete_category($request) {
        $taxonomy = $this->resolve_taxonomy($request->get_param('taxonomy'));
        $term_id = intval($request->get_param('id'));

        if (!$term_id) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => __('Invalid category ID', 'mosallas')
            ), 400);
        }

        $existing_term = get_term($term_id, $taxonomy);
        if (!$existing_term || is_wp_error($existing_term)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => __('Category not found', 'mosallas')
            ), 404);
        }

        $reassign_param = $request->get_param('reassign') ?: $request->get_param('reassignTo');
        $reassign_id = $reassign_param ? intval($reassign_param) : intval(get_option('default_category'));

        if ($reassign_id === $term_id) {
            $reassign_id = intval(get_option('default_category'));
        }

        $args = array();
        if ($reassign_id > 0) {
            $args['default'] = $reassign_id;
        }

        $result = wp_delete_term($term_id, $taxonomy, $args);

        if (is_wp_error($result)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => $result->get_error_message()
            ), 400);
        }

        if (!$result) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => __('Failed to delete category', 'mosallas')
            ), 500);
        }

        return new WP_REST_Response(array(
            'success' => true,
            'deleted' => true,
            'reassignedTo' => $reassign_id > 0 ? $reassign_id : null
        ), 200);
    }

    /**
     * Normalize category payload
     */
    private function format_category($term, $taxonomy) {
        return array(
            'id' => intval($term->term_id),
            'name' => $term->name,
            'slug' => $term->slug,
            'description' => $term->description,
            'parentId' => intval($term->parent),
            'count' => intval($term->count),
            'taxonomy' => $taxonomy,
            'link' => get_term_link($term)
        );
    }

    /**
     * Restrict taxonomy to supported ones
     */
    private function resolve_taxonomy($input) {
        $allowed = array('category', 'product_cat');
        $taxonomy = sanitize_text_field($input);

        if (empty($taxonomy) || !in_array($taxonomy, $allowed, true)) {
            return 'category';
        }

        return $taxonomy;
    }
}

