<?php
/**
 * Mosallas llms.txt Generator
 * Serves /llms.txt and (optionally) writes it to the site root.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mosallas_LLMs_Txt {

    const CACHE_OPTION = 'mosallas_llms_txt_cached';
    const CACHE_AT_OPTION = 'mosallas_llms_txt_cached_at';
    const CACHE_TTL_SECONDS = 86400; // 24h
    const GEMINI_MODEL = 'models/gemini-flash-lite-latest';

    const TEMPLATE = "# llms.txt for YourSiteName\n> A concise summary of what your site offers. Include core value or unique focus.\n\n## Key Pages\n- [Home](https://yoursite.com) – Main overview and entry point\n- [About](https://yoursite.com/about/) – Who we are and mission\n- [Services](https://yoursite.com/services/) – Core offerings\n- [Products](https://yoursite.com/products/) – Full product catalog\n- [Blog](https://yoursite.com/blog/) – Helpful articles and insights\n- [Contact](https://yoursite.com/contact/) – Ways to reach us\n\n## FAQ / Useful Resources\n- [FAQ Section](https://yoursite.com/faq/) – Common questions answered\n- [Support](https://yoursite.com/support/) – Help / documentation\n";

    /**
     * Boot hooks
     */
    public function init() {
        // Serve llms.txt early and without requiring rewrite rules.
        add_action('template_redirect', array($this, 'maybe_serve_llms_txt'), 0);
    }

    /**
     * Serve llms.txt if requested.
     */
    public function maybe_serve_llms_txt() {
        $request_uri = isset($_SERVER['REQUEST_URI']) ? sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI'])) : '';
        if (empty($request_uri)) {
            return;
        }

        $path = wp_parse_url($request_uri, PHP_URL_PATH);
        if ($path === null) {
            return;
        }

        // Match both /llms.txt and /subdir/llms.txt.
        if (!preg_match('~(^|/)llms\\.txt$~i', $path)) {
            return;
        }

        $content = self::get_llms_txt();

        status_header(200);
        header('Content-Type: text/plain; charset=utf-8');
        header('X-Robots-Tag: noindex');
        echo $content;
        exit;
    }

    /**
     * Get llms.txt from cache or refresh via webhook.
     */
    public static function get_llms_txt() {
        $cached = get_option(self::CACHE_OPTION, '');
        $cached_at = (int) get_option(self::CACHE_AT_OPTION, 0);

        if (!empty($cached) && $cached_at > 0 && (time() - $cached_at) < self::CACHE_TTL_SECONDS) {
            return $cached;
        }

        $refresh = self::refresh_llms_txt_via_webhook();
        if (!empty($refresh['success']) && !empty($refresh['content'])) {
            return $refresh['content'];
        }

        // Fallback to deterministic local generator.
        return self::generate_llms_txt_fallback();
    }

    /**
     * Refresh llms.txt via Mosallas webhook (server-side will call Gemini).
     *
     * @return array{success:bool,content:string,message:string,http_status?:int}
     */
    public static function refresh_llms_txt_via_webhook() {
        $api_key = get_option('mosallas_api_key');
        if (empty($api_key)) {
            return array('success' => false, 'content' => '', 'message' => 'API key not configured');
        }

        $webhook_url = get_option('mosallas_webhook_url', 'https://mosallas.ir/api/wordpress/webhook');
        if (empty($webhook_url)) {
            $webhook_url = 'https://mosallas.ir/api/wordpress/webhook';
        }

        $site_url = get_site_url();
        $site_name = get_bloginfo('name');
        $site_id = get_option('mosallas_site_id');
        $project_id = get_option('mosallas_project_id');
        $system_prompt = get_option('mosallas_project_system_prompt', '');

        $payload = array(
            'action' => 'generate_llms_txt',
            'data' => array(
                'provider' => 'gemini',
                'model' => self::GEMINI_MODEL,
                'template' => self::TEMPLATE,
                'site' => array(
                    'siteId' => $site_id,
                    'projectId' => $project_id,
                    'siteUrl' => $site_url,
                    'siteName' => $site_name,
                    'siteDescription' => get_bloginfo('description'),
                ),
                'project' => array(
                    'systemPrompt' => $system_prompt,
                ),
                'internalLinks' => self::collect_internal_links_payload(),
            ),
        );

        $response = wp_remote_post($webhook_url, array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode($payload),
            'timeout' => 20,
        ));

        if (is_wp_error($response)) {
            return array('success' => false, 'content' => '', 'message' => $response->get_error_message());
        }

        $http_status = (int) wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        $content = '';
        if (is_array($body)) {
            if (!empty($body['llmsTxt'])) {
                $content = (string) $body['llmsTxt'];
            } elseif (!empty($body['content'])) {
                $content = (string) $body['content'];
            } elseif (!empty($body['data']['llmsTxt'])) {
                $content = (string) $body['data']['llmsTxt'];
            }
        }

        $content = self::normalize_llms_txt($content);
        if (empty($content)) {
            return array(
                'success' => false,
                'content' => '',
                'message' => 'Webhook returned empty llms.txt',
                'http_status' => $http_status,
            );
        }

        update_option(self::CACHE_OPTION, $content, false);
        update_option(self::CACHE_AT_OPTION, time(), false);

        return array(
            'success' => true,
            'content' => $content,
            'message' => 'llms.txt refreshed via webhook',
            'http_status' => $http_status,
        );
    }

    /**
     * Generate llms.txt content.
     */
    public static function generate_llms_txt_fallback() {
        $site_name = get_bloginfo('name');
        if (empty($site_name)) {
            $site_name = wp_parse_url(home_url('/'), PHP_URL_HOST);
        }
        if (empty($site_name)) {
            $site_name = 'YourSiteName';
        }

        $summary_source = get_option('mosallas_project_system_prompt', '');
        if (empty($summary_source)) {
            $summary_source = get_bloginfo('description');
        }

        $summary = self::summarize_text($summary_source);
        if (empty($summary)) {
            $summary = 'A concise summary of what your site offers.';
        }

        $home_url = home_url('/');

        $key_pages = array(
            array('label' => 'Home', 'url' => $home_url, 'desc' => 'Main overview and entry point'),
            array('label' => 'About', 'url' => self::resolve_internal_url(array('about', 'about-us', 'aboutus')), 'desc' => 'Who we are and mission'),
            array('label' => 'Services', 'url' => self::resolve_internal_url(array('services', 'service')), 'desc' => 'Core offerings'),
            array('label' => 'Products', 'url' => self::resolve_products_url(), 'desc' => 'Full product catalog'),
            array('label' => 'Blog', 'url' => self::resolve_blog_url(), 'desc' => 'Helpful articles and insights'),
            array('label' => 'Contact', 'url' => self::resolve_internal_url(array('contact', 'contact-us', 'contactus')), 'desc' => 'Ways to reach us'),
        );

        $resources = array(
            array('label' => 'FAQ Section', 'url' => self::resolve_internal_url(array('faq', 'faqs')), 'desc' => 'Common questions answered'),
            array('label' => 'Support', 'url' => self::resolve_internal_url(array('support', 'help', 'docs', 'documentation')), 'desc' => 'Help / documentation'),
        );

        $lines = array();
        $lines[] = '# llms.txt for ' . self::normalize_single_line($site_name);
        $lines[] = '> ' . self::normalize_single_line($summary);
        $lines[] = '';
        $lines[] = '## Key Pages';
        foreach ($key_pages as $item) {
            $lines[] = '- [' . $item['label'] . '](' . esc_url($item['url']) . ') – ' . $item['desc'];
        }
        $lines[] = '';
        $lines[] = '## FAQ / Useful Resources';
        foreach ($resources as $item) {
            $lines[] = '- [' . $item['label'] . '](' . esc_url($item['url']) . ') – ' . $item['desc'];
        }
        $lines[] = '';

        return implode("\n", $lines);
    }

    /**
     * Attempt to write llms.txt to site root.
     *
     * @return array{success:bool,path:string,message:string}
     */
    public static function write_llms_txt_to_root() {
        $content = self::get_llms_txt();

        if (!function_exists('get_home_path')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        $root = function_exists('get_home_path') ? get_home_path() : ABSPATH;
        $path = trailingslashit($root) . 'llms.txt';

        // Prefer WP_Filesystem when available.
        if (!function_exists('WP_Filesystem')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        $fs_ready = WP_Filesystem();
        if ($fs_ready) {
            global $wp_filesystem;
            if ($wp_filesystem && method_exists($wp_filesystem, 'put_contents')) {
                $ok = $wp_filesystem->put_contents($path, $content, FS_CHMOD_FILE);
                if ($ok) {
                    return array('success' => true, 'path' => $path, 'message' => 'llms.txt written via WP_Filesystem');
                }
            }
        }

        // Fallback to direct write if possible.
        $dir = dirname($path);
        if (is_dir($dir) && is_writable($dir)) {
            $written = @file_put_contents($path, $content);
            if ($written !== false) {
                return array('success' => true, 'path' => $path, 'message' => 'llms.txt written via file_put_contents');
            }
        }

        return array('success' => false, 'path' => $path, 'message' => 'Unable to write llms.txt to site root');
    }

    /**
     * Create a structured internal links payload for the webhook.
     */
    private static function collect_internal_links_payload() {
        $items = array();

        $items[] = array('label' => 'Home', 'url' => home_url('/'), 'type' => 'page');

        $known_pages = array(
            array('label' => 'About', 'slugs' => array('about', 'about-us', 'aboutus'), 'type' => 'page'),
            array('label' => 'Services', 'slugs' => array('services', 'service'), 'type' => 'page'),
            array('label' => 'Contact', 'slugs' => array('contact', 'contact-us', 'contactus'), 'type' => 'page'),
            array('label' => 'FAQ', 'slugs' => array('faq', 'faqs'), 'type' => 'page'),
            array('label' => 'Support', 'slugs' => array('support', 'help', 'docs', 'documentation'), 'type' => 'page'),
        );

        foreach ($known_pages as $p) {
            $url = self::resolve_internal_url($p['slugs']);
            if (!empty($url)) {
                $items[] = array('label' => $p['label'], 'url' => $url, 'type' => $p['type']);
            }
        }

        $items[] = array('label' => 'Blog', 'url' => self::resolve_blog_url(), 'type' => 'archive');
        $items[] = array('label' => 'Products', 'url' => self::resolve_products_url(), 'type' => 'archive');

        // Add a handful of recent posts/pages as internal links.
        $recent_posts = get_posts(array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'numberposts' => 20,
            'orderby' => 'date',
            'order' => 'DESC',
            'suppress_filters' => true,
        ));
        foreach ($recent_posts as $post) {
            $permalink = get_permalink($post);
            if (!empty($permalink)) {
                $items[] = array('label' => get_the_title($post), 'url' => $permalink, 'type' => 'post');
            }
        }

        $recent_pages = get_pages(array(
            'post_status' => 'publish',
            'number' => 20,
            'sort_column' => 'post_date',
            'sort_order' => 'desc',
        ));
        foreach ($recent_pages as $page) {
            $permalink = get_permalink($page);
            if (!empty($permalink)) {
                $items[] = array('label' => get_the_title($page), 'url' => $permalink, 'type' => 'page');
            }
        }

        // De-dup by URL.
        $seen = array();
        $deduped = array();
        foreach ($items as $item) {
            $url = isset($item['url']) ? (string) $item['url'] : '';
            if (empty($url)) {
                continue;
            }
            if (isset($seen[$url])) {
                continue;
            }
            $seen[$url] = true;
            $deduped[] = $item;
        }

        return $deduped;
    }

    private static function normalize_llms_txt($content) {
        $content = is_string($content) ? $content : '';
        $content = str_replace("\r\n", "\n", $content);
        $content = str_replace("\r", "\n", $content);
        $content = trim($content);

        // Strip non-printable control chars except newline and tab.
        $content = preg_replace('/[\x00-\x08\x0B\x0C\x0E-\x1F\x7F]/u', '', $content);

        return $content;
    }

    private static function resolve_internal_url($candidates) {
        $candidates = is_array($candidates) ? $candidates : array();

        foreach ($candidates as $slug) {
            $slug = sanitize_title($slug);
            if (empty($slug)) {
                continue;
            }

            $page = get_page_by_path($slug);
            if ($page && !empty($page->ID)) {
                $link = get_permalink($page->ID);
                if (!empty($link)) {
                    return $link;
                }
            }
        }

        // Fallback: use the first candidate as a conventional path.
        $fallback = !empty($candidates[0]) ? sanitize_title($candidates[0]) : '';
        if (!empty($fallback)) {
            return home_url('/' . $fallback . '/');
        }

        return home_url('/');
    }

    private static function resolve_blog_url() {
        $page_for_posts = (int) get_option('page_for_posts');
        if (!empty($page_for_posts)) {
            $link = get_permalink($page_for_posts);
            if (!empty($link)) {
                return $link;
            }
        }

        return home_url('/blog/');
    }

    private static function resolve_products_url() {
        if (function_exists('wc_get_page_permalink')) {
            $shop = wc_get_page_permalink('shop');
            if (!empty($shop)) {
                return $shop;
            }
        }

        $archive = function_exists('get_post_type_archive_link') ? get_post_type_archive_link('product') : '';
        if (!empty($archive)) {
            return $archive;
        }

        return self::resolve_internal_url(array('products', 'shop'));
    }

    private static function summarize_text($text) {
        $text = is_string($text) ? $text : '';
        $text = wp_strip_all_tags($text);
        $text = trim(preg_replace('/\s+/', ' ', $text));

        if ($text === '') {
            return '';
        }

        // Take first sentence if it is reasonably short, else truncate.
        $first_sentence = preg_split('/(?<=[\.\!\?])\s+/', $text, 2);
        if (is_array($first_sentence) && !empty($first_sentence[0])) {
            $candidate = trim($first_sentence[0]);
            if (mb_strlen($candidate) <= 240) {
                return $candidate;
            }
        }

        if (mb_strlen($text) <= 240) {
            return $text;
        }

        return rtrim(mb_substr($text, 0, 237)) . '...';
    }

    private static function normalize_single_line($text) {
        $text = is_string($text) ? $text : '';
        $text = trim(preg_replace('/\s+/', ' ', $text));
        return $text;
    }
}
