<?php
/**
 * Mosallas Publisher
 * Handles article creation and publishing
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mosallas_Publisher {
    
    /**
     * Store current article data for category selection
     */
    private $current_article_title = '';
    private $current_article_content = '';
    private $current_article_id = '';
    private $current_faq_schema = null;
    
    /**
     * Create and publish a post
     */
    public function create_post($data) {
        try {
            error_log('Mosallas Publisher: Starting post creation');
            error_log('Mosallas Publisher: Data keys - ' . implode(', ', array_keys($data)));
            
            // Store article data for category selection
            $this->current_article_title = $data['title'] ?? '';
            $this->current_article_content = $data['content'] ?? '';
            $this->current_article_id = $data['articleId'] ?? '';
            
            // Validate input data
            if (!is_array($data)) {
                throw new Exception('داده‌های ورودی نامعتبر است');
            }
            
            if (empty($data['title'])) {
                throw new Exception('عنوان مقاله خالی است');
            }
            
            if (empty($data['content'])) {
                throw new Exception('محتوای مقاله خالی است');
            }
            
            $featured_image_candidates = $this->get_featured_image_candidates($data);
            
            // Get or create category
            try {
                $category_ids = $this->get_or_create_category($data['topic'] ?? '');
                error_log('Mosallas Publisher: Category IDs: ' . print_r($category_ids, true));
                Mosallas_Logger::info('📁 دسته‌بندی تنظیم شد', array(
                    'category_ids' => $category_ids,
                    'topic' => $data['topic'] ?? ''
                ));
            } catch (Exception $cat_error) {
                error_log('Mosallas Publisher: Category creation failed - ' . $cat_error->getMessage());
                Mosallas_Logger::warning('⚠️ خطا در ایجاد دسته‌بندی (استفاده از دسته پیش‌فرض)', array(
                    'error' => $cat_error->getMessage()
                ));
                $category_ids = array(get_option('default_category', 1));
            }
            
            // Format the content
            try {
                error_log('Mosallas Publisher: Formatting content');
                Mosallas_Logger::info('📝 در حال قالب‌بندی محتوا');
                $formatted_content = $this->format_content($data);
            } catch (Exception $format_error) {
                error_log('Mosallas Publisher: Content formatting failed - ' . $format_error->getMessage());
                Mosallas_Logger::error('❌ خطا در قالب‌بندی محتوا', array(
                    'error' => $format_error->getMessage()
                ));
                throw new Exception('خطا در قالب‌بندی محتوا: ' . $format_error->getMessage());
            }
            
            // Prepare post data
            $post_data = array(
                'post_title' => sanitize_text_field($data['title']),
                'post_content' => $formatted_content,
                'post_excerpt' => !empty($data['abstract']) ? sanitize_textarea_field($data['abstract']) : '',
                'post_status' => 'publish',
                'post_author' => $this->get_author_id(),
                'post_category' => $category_ids,
                'post_type' => 'post'
            );

            // Set post slug if provided
            if (!empty($data['slug'])) {
                $post_data['post_name'] = sanitize_title($data['slug']);
            }
            
            error_log('Mosallas Publisher: Inserting post');
            Mosallas_Logger::info('💾 در حال ذخیره مقاله در وردپرس', array(
                'title' => $data['title']
            ));
            
            // Insert the post
            $post_id = wp_insert_post($post_data, true);
            
            if (is_wp_error($post_id)) {
                $error_message = $post_id->get_error_message();
                error_log('Mosallas Publisher: wp_insert_post failed - ' . $error_message);
                Mosallas_Logger::error('❌ خطا در ذخیره مقاله در پایگاه داده وردپرس', array(
                    'error' => $error_message,
                    'post_data' => $post_data
                ));
                throw new Exception('خطا در ذخیره مقاله: ' . $error_message);
            }
            
            error_log('Mosallas Publisher: Post created with ID: ' . $post_id);
            Mosallas_Logger::success('✅ مقاله در وردپرس ذخیره شد', array(
                'post_id' => $post_id
            ));
            
            $featured_image_id = 0;
            
            // Priority 1: Use featuredImageUrl or featured_image_url from payload
            $payload_featured_image_url = '';
            if (!empty($data['featuredImageUrl'])) {
                $payload_featured_image_url = $data['featuredImageUrl'];
            } elseif (!empty($data['featured_image_url'])) {
                $payload_featured_image_url = $data['featured_image_url'];
            }
            
            if (!empty($payload_featured_image_url)) {
                error_log('Mosallas Publisher: Using payload featured image URL: ' . $payload_featured_image_url);
                Mosallas_Logger::info('🖼️ استفاده از تصویر شاخص ارسال‌شده در payload');
                $featured_image_id = $this->assign_featured_image($post_id, array($payload_featured_image_url), $data['title']);
            }
            
            // Priority 2: If no payload image, try other candidates from data
            if (!$featured_image_id) {
                $featured_image_candidates = $this->get_featured_image_candidates($data);
                if (!empty($featured_image_candidates)) {
                    error_log('Mosallas Publisher: Using other featured image candidates from data');
                    Mosallas_Logger::info('🖼️ استفاده از سایر کاندیداهای تصویر شاخص');
                    $featured_image_id = $this->assign_featured_image($post_id, $featured_image_candidates, $data['title']);
                }
            }

            // Priority 3: Fallback to extracting images from article content
            if (!$featured_image_id && !empty($data['content'])) {
                error_log('Mosallas Publisher: No payload or data images found, extracting from content');
                Mosallas_Logger::info('🖼️ استخراج تصویر از محتوای مقاله');
                $fallback_candidates = $this->extract_image_sources_from_content($data['content']);
                if (!empty($fallback_candidates)) {
                    $featured_image_id = $this->assign_featured_image($post_id, $fallback_candidates, $data['title']);
                }
            }
            
            // Update SEO meta tags
            try {
                error_log('Mosallas Publisher: Updating SEO meta tags');
                Mosallas_Logger::info('🎯 در حال تنظیم تگ‌های سئو');
                $this->update_seo_meta($post_id, $data);
                Mosallas_Logger::success('✅ تگ‌های سئو تنظیم شدند');
            } catch (Exception $seo_error) {
                error_log('Mosallas Publisher: SEO meta update failed - ' . $seo_error->getMessage());
                Mosallas_Logger::warning('⚠️ خطا در تنظیم تگ‌های سئو', array(
                    'error' => $seo_error->getMessage()
                ));
            }

            // Persist FAQ schema separately for head injection
            $this->persist_faq_schema($post_id, $this->current_faq_schema);
            
            // Add internal links (after post is created)
            if (!empty($data['existingPosts'])) {
                error_log('Mosallas Publisher: Adding internal links');
                $this->add_internal_links($post_id, $data['existingPosts']);
            }
            
            // Add custom meta data
            error_log('Mosallas Publisher: Adding custom meta data');
            update_post_meta($post_id, '_mosallas_generated', true);
            update_post_meta($post_id, '_mosallas_generated_date', current_time('mysql'));
            if (!empty($data['articleId'])) {
                update_post_meta($post_id, '_mosallas_article_id', sanitize_text_field($data['articleId']));
            }
            
            error_log('Mosallas Publisher: Post creation completed successfully');
            
            return $post_id;
            
        } catch (Exception $e) {
            error_log('Mosallas Publisher: Exception caught - ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Update an existing WordPress post with new content received from Mosallas.ir
     */
    public function update_existing_post($post_id, $data) {
        if (!is_array($data)) {
            throw new Exception('داده‌های ورودی نامعتبر است');
        }

        $post = get_post($post_id);
        if (!$post) {
            throw new Exception('مقاله مورد نظر در وردپرس یافت نشد');
        }

        Mosallas_Logger::info('🛠️ درخواست بروزرسانی دستی مقاله دریافت شد', array(
            'post_id' => $post_id,
            'article_id' => $data['articleId'] ?? 'unknown'
        ));

        $this->current_article_title = isset($data['title']) ? $data['title'] : $post->post_title;
        $this->current_article_content = isset($data['content']) ? $data['content'] : $post->post_content;
        $this->current_article_id = $data['articleId'] ?? get_post_meta($post_id, '_mosallas_article_id', true);

        $skip_formatting = !empty($data['skipFormatting']);
        $content_provided = array_key_exists('content', $data);

        // Capture FAQ schema if provided during updates
        $this->current_faq_schema = $this->normalize_faq_schema($data['faqSchema'] ?? null);

        $update_data = array('ID' => $post_id);

        if (array_key_exists('title', $data)) {
            $update_data['post_title'] = sanitize_text_field($data['title']);
        }

        if (!empty($data['slug'])) {
            $update_data['post_name'] = sanitize_title($data['slug']);
        }

        if (array_key_exists('abstract', $data)) {
            $update_data['post_excerpt'] = sanitize_textarea_field($data['abstract']);
        }

        if ($content_provided) {
            if ($skip_formatting) {
                $update_data['post_content'] = wp_kses_post($data['content']);
            } else {
                $update_data['post_content'] = $this->format_content($data);
            }
        }

        if (array_key_exists('status', $data)) {
            $allowed_statuses = array('draft', 'pending', 'publish', 'private');
            $status_value = strtolower((string) $data['status']);
            if (in_array($status_value, $allowed_statuses, true)) {
                $update_data['post_status'] = $status_value;
            }
        }

        $result = wp_update_post($update_data, true);

        if (is_wp_error($result)) {
            $error_message = $result->get_error_message();
            Mosallas_Logger::error('❌ بروزرسانی مقاله ناموفق بود', array(
                'post_id' => $post_id,
                'error' => $error_message
            ));
            throw new Exception('خطا در بروزرسانی مقاله: ' . $error_message);
        }

        Mosallas_Logger::success('✏️ مقاله با موفقیت بروزرسانی شد', array(
            'post_id' => $post_id
        ));

        if (!empty($data['categoryIds']) && is_array($data['categoryIds'])) {
            $category_ids = array_map('intval', $data['categoryIds']);
            wp_set_post_categories($post_id, $category_ids);
        } elseif (!empty($data['topic']) && is_string($data['topic'])) {
            try {
                $category_ids = $this->get_or_create_category($data['topic']);
                wp_set_post_categories($post_id, $category_ids);
            } catch (Exception $category_error) {
                Mosallas_Logger::warning('⚠️ بروزرسانی دسته‌بندی ناموفق بود', array(
                    'post_id' => $post_id,
                    'error' => $category_error->getMessage()
                ));
            }
        }

        if (!empty($data['removeFeaturedImage'])) {
            delete_post_thumbnail($post_id);
        }

        $featured_candidates = $this->get_featured_image_candidates($data);
        if (!empty($featured_candidates)) {
            $this->assign_featured_image($post_id, $featured_candidates, $data['title'] ?? $post->post_title);
        }

        if (!empty($data['existingPosts'])) {
            $this->add_internal_links($post_id, $data['existingPosts']);
        }

        $this->update_seo_meta($post_id, $data);

        // Persist FAQ schema separately for head injection
        $this->persist_faq_schema($post_id, $this->current_faq_schema ?? ($data['faqSchema'] ?? null));

        if (!empty($data['articleId'])) {
            update_post_meta($post_id, '_mosallas_article_id', sanitize_text_field($data['articleId']));
        }

        update_post_meta($post_id, '_mosallas_generated', true);
        update_post_meta($post_id, '_mosallas_generated_date', current_time('mysql'));

        $thumb_id = get_post_thumbnail_id($post_id);
        $thumb_url = '';
        if ($thumb_id) {
            $src = wp_get_attachment_image_src($thumb_id, 'full');
            if ($src && is_array($src)) {
                $thumb_url = $src[0];
            }
        }

        return array(
            'post_id' => $post_id,
            'post_url' => get_permalink($post_id),
            'edit_url' => get_edit_post_link($post_id, 'raw'),
            'featuredImageId' => $thumb_id,
            'featuredImageUrl' => $thumb_url
        );
    }

    /**
     * Resolve ordered candidate URLs for the featured image.
     */
    private function get_featured_image_candidates($data) {
        $candidates = array();

        if (!is_array($data)) {
            return $candidates;
        }

        $keys = array(
            'featuredImageUrl',
            'featured_image_url',
            'featuredImageURL',
            'featured_image',
            'featuredImage',
            'coverImageUrl',
            'cover_image_url'
        );

        foreach ($keys as $key) {
            if (!isset($data[$key])) {
                continue;
            }

            $value = $data[$key];

            if (is_string($value) && !empty(trim($value))) {
                $candidates[] = trim($value);
            } elseif (is_array($value) && isset($value['url']) && !empty($value['url'])) {
                $candidates[] = trim($value['url']);
            }
        }

        // Flatten nested image arrays that may be provided
        if (isset($data['featuredImage']) && is_array($data['featuredImage'])) {
            if (isset($data['featuredImage']['src'])) {
                $candidates[] = trim($data['featuredImage']['src']);
            }
            if (isset($data['featuredImage']['source'])) {
                $candidates[] = trim($data['featuredImage']['source']);
            }
        }

        $unique = array();
        $final = array();
        foreach ($candidates as $candidate) {
            if (!is_string($candidate) || $candidate === '') {
                continue;
            }

            $candidate = html_entity_decode(trim($candidate));
            if ($candidate === '') {
                continue;
            }

            if (!preg_match('/^https?:\/\//i', $candidate)) {
                continue; // Ignore relative URLs
            }

            $hash = md5($candidate);
            if (isset($unique[$hash])) {
                continue;
            }

            $unique[$hash] = true;
            $final[] = $candidate;
        }

        return $final;
    }

    /**
     * Try assigning a featured image using the provided candidate URLs.
     */
    private function assign_featured_image($post_id, array $candidates, $title) {
        foreach ($candidates as $candidate_url) {
            try {
                $attachment_id = $this->upload_image($candidate_url, $title, $post_id);
                if ($attachment_id > 0) {
                    // Set the uploaded image as the post's featured image
                    $result = set_post_thumbnail($post_id, $attachment_id);
                    if ($result) {
                        Mosallas_Logger::success('🖼️ تصویر شاخص به مقاله اضافه شد', array(
                            'attachment_id' => $attachment_id,
                            'post_id' => $post_id,
                            'source' => $candidate_url
                        ));
                        return $attachment_id;
                    } else {
                        Mosallas_Logger::warning('⚠️ تصویر آپلود شد اما تنظیم به عنوان تصویر شاخص ناموفق بود', array(
                            'attachment_id' => $attachment_id,
                            'post_id' => $post_id,
                            'source' => $candidate_url
                        ));
                    }
                }
            } catch (Exception $e) {
                Mosallas_Logger::warning('⚠️ تلاش برای تنظیم تصویر شاخص ناموفق بود', array(
                    'error' => $e->getMessage(),
                    'source' => $candidate_url
                ));
            }
        }

        return 0;
    }

    /**
     * Extract image sources from article HTML content.
     */
    private function extract_image_sources_from_content($content) {
        $sources = array();

        if (empty($content) || !is_string($content)) {
            return $sources;
        }

        $dom = new DOMDocument();
        libxml_use_internal_errors(true);
        $loaded = $dom->loadHTML(mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8'));
        libxml_clear_errors();

        if (!$loaded) {
            return $sources;
        }

        $images = $dom->getElementsByTagName('img');
        foreach ($images as $image) {
            $src = $image->getAttribute('src');
            if (empty($src)) {
                continue;
            }

            $src = html_entity_decode(trim($src));
            if (!preg_match('/^https?:\/\//i', $src)) {
                continue;
            }

            $sources[] = $src;
        }

        return $sources;
    }
    
    /**
     * Format article content with all sections
     */
    private function format_content($data) {
        try {
            $content = '';
            
            // Validate data is not null
            if (!is_array($data)) {
                error_log('Mosallas Publisher: Invalid data format for format_content - not an array');
                throw new Exception('داده‌های مقاله نامعتبر است');
            }

            // Capture FAQ schema if provided separately
            $this->current_faq_schema = $this->normalize_faq_schema($data['faqSchema'] ?? null);
            
            // Add abstract/introduction as clean HTML
            if (!empty($data['abstract'])) {
                $content .= "\n\n" . '<p><strong>خلاصه:</strong></p>' . "\n";
                $content .= '<p>' . wp_kses_post($data['abstract']) . '</p>' . "\n\n";
            }
            
            // Add audio summary player if available (PRO feature)
            // Validate that audioUrl is a proper URL before adding the player
            if (!empty($data['audioUrl']) && is_string($data['audioUrl']) && filter_var($data['audioUrl'], FILTER_VALIDATE_URL)) {
                $audio_url = esc_url($data['audioUrl']);
                // Double-check the escaped URL is not empty
                if (!empty($audio_url) && strlen($audio_url) > 10) {
                    // Minimal design: simple border, light background, neutral colors
                    // Force text color to dark gray to ensure visibility on dark themes
                    $content .= '<div class="mosallas-audio-summary" style="border: 1px solid #e5e7eb; background-color: #f9fafb; padding: 16px; border-radius: 8px; margin: 24px 0; color: #1f2937;">' . "\n";
                    $content .= '<p style="margin: 0 0 12px 0; font-weight: 600; display: flex; align-items: center; gap: 8px; color: #1f2937;">' . "\n";
                    $content .= '<span style="font-size: 1.2em;">🎧</span> خلاصه صوتی مقاله' . "\n";
                    $content .= '</p>' . "\n";
                    // Use src attribute directly on audio tag for better compatibility
                    $content .= '<audio controls src="' . $audio_url . '" style="width: 100%;">' . "\n";
                    $content .= 'مرورگر شما از پخش صوت پشتیبانی نمی‌کند.' . "\n";
                    $content .= '</audio>' . "\n";
                    $content .= '</div>' . "\n\n";
                    error_log('Mosallas Publisher: Added audio summary player - ' . $audio_url);
                } else {
                    error_log('Mosallas Publisher: Skipping audio player - invalid escaped URL');
                }
            } else if (!empty($data['audioUrl'])) {
                error_log('Mosallas Publisher: Skipping audio player - audioUrl is not a valid URL: ' . print_r($data['audioUrl'], true));
            }
            
            // Add table of contents as clean HTML
            if (!empty($data['tableOfContents'])) {
                $toc = $data['tableOfContents'];
                
                // If it's a JSON string, decode it
                if (is_string($toc)) {
                    $decoded = json_decode($toc, true);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $toc = $decoded;
                    } else {
                        error_log('Mosallas Publisher: JSON decode error for tableOfContents - ' . json_last_error_msg());
                        $toc = array(); // Reset to empty array on error
                    }
                }
                
                if (is_array($toc) && !empty($toc)) {
                    $content .= '<h2>فهرست مطالب</h2>' . "\n";
                    $content .= '<ul>' . "\n";
                    foreach ($toc as $item) {
                        $heading = '';
                        $id = '';

                        if (is_string($item)) {
                            $heading = $item;
                        } elseif (is_array($item)) {
                            // Support structures such as { title: "...", id: "..." }
                            $heading = $item['title'] ?? $item['text'] ?? $item['label'] ?? '';
                            $id = $item['id'] ?? $item['anchor'] ?? '';
                        } elseif (is_object($item)) {
                            $heading = $item->title ?? $item->text ?? $item->label ?? '';
                            $id = isset($item->id) ? $item->id : (isset($item->anchor) ? $item->anchor : '');
                        }

                        if (!empty($heading)) {
                            if (!empty($id)) {
                                $content .= '<li><a href="#' . esc_attr($id) . '">' . esc_html($heading) . '</a></li>' . "\n";
                            } else {
                                $content .= '<li>' . esc_html($heading) . '</li>' . "\n";
                            }
                        }
                    }
                    $content .= '</ul>' . "\n\n";
                }
            }
            
            // Add main content as raw HTML (this is the main article content)
            if (!empty($data['content'])) {
                // Use the content as-is, but allow WordPress to handle it properly
                // Remove excessive filtering to preserve HTML structure for editing
                $main_content = $data['content'];
                
                // Only do basic security filtering, preserve HTML structure
                $allowed_html = wp_kses_allowed_html('post');
                
                // Add additional allowed tags for better content preservation
                $allowed_html['figure'] = array(
                    'class' => array(),
                    'id' => array(),
                    'style' => array()
                );
                $allowed_html['figcaption'] = array(
                    'class' => array(),
                    'id' => array()
                );
                $allowed_html['section'] = array(
                    'class' => array(),
                    'id' => array()
                );
                $allowed_html['article'] = array(
                    'class' => array(),
                    'id' => array()
                );
                $allowed_html['audio'] = array(
                    'controls' => array(),
                    'src' => array(),
                    'style' => array(),
                    'class' => array(),
                    'id' => array(),
                    'preload' => array()
                );
                $allowed_html['source'] = array(
                    'src' => array(),
                    'type' => array()
                );
                
                $content .= wp_kses($main_content, $allowed_html) . "\n\n";
            } else {
                error_log('Mosallas Publisher: Missing content field');
                throw new Exception('محتوای مقاله خالی است');
            }
            
            // Add FAQ section as clean HTML
            if (!empty($data['faqSection'])) {
                $faqContent = $data['faqSection'];

                // Extract embedded JSON-LD from FAQ content (if any) and keep it for head injection
                $faqContent = $this->strip_and_capture_faq_schema($faqContent, $this->current_faq_schema);

                // If FAQ is provided as JSON, transform to HTML
                if (is_string($faqContent)) {
                    $decodedFaq = json_decode($faqContent, true);
                    if (json_last_error() === JSON_ERROR_NONE && is_array($decodedFaq)) {
                        $faqHtml = '';
                        foreach ($decodedFaq as $faqItem) {
                            if (is_array($faqItem)) {
                                $question = $faqItem['question'] ?? '';
                                $answer = $faqItem['answer'] ?? '';
                            } elseif (is_object($faqItem)) {
                                $question = $faqItem->question ?? '';
                                $answer = $faqItem->answer ?? '';
                            } else {
                                $question = '';
                                $answer = '';
                            }

                            if (!empty($question) || !empty($answer)) {
                                if (!empty($question)) {
                                    $faqHtml .= '<h3>' . esc_html($question) . '</h3>' . "\n";
                                }
                                if (!empty($answer)) {
                                    $faqHtml .= '<p>' . wp_kses_post($answer) . '</p>' . "\n";
                                }
                            }
                        }

                        if (!empty($faqHtml)) {
                            $faqContent = $faqHtml;
                        }
                    }
                }

                $content .= '<h2>سوالات متداول (FAQ)</h2>' . "\n";
                $content .= wp_kses_post($faqContent) . "\n";
            }
            
            return $content;
            
        } catch (Exception $e) {
            error_log('Mosallas Publisher: Exception in format_content - ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Upload image from URL to WordPress media library
     */
    private function upload_image($image_url, $title, $post_id = 0) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        $image_url = $this->normalize_remote_image_url($image_url);
        if (empty($image_url)) {
            Mosallas_Logger::warning('⚠️ آدرس تصویر شاخص نامعتبر است');
            return 0;
        }

        // Download the image
        $temp_file = download_url($image_url, 30);
        
        if (is_wp_error($temp_file)) {
            error_log('Mosallas: Failed to download image - ' . $temp_file->get_error_message());
            Mosallas_Logger::warning('⚠️ دانلود تصویر شاخص ناموفق بود', array(
                'error' => $temp_file->get_error_message(),
                'source' => $image_url
            ));
            return 0;
        }
        
        // Try to resolve file extension from URL path (ignoring query params)
        $parsed_url = wp_parse_url($image_url);
        $url_path = isset($parsed_url['path']) ? $parsed_url['path'] : '';
        $file_extension = strtolower(pathinfo($url_path, PATHINFO_EXTENSION));

        // Initial sanitization to drop unexpected characters
        if (!empty($file_extension)) {
            $file_extension = preg_replace('/[^a-z0-9]/', '', $file_extension);
        }

        $mime_type = '';
        $allowed_extensions = array('jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp', 'tiff', 'svg', 'avif', 'heic', 'heif');

        if (empty($file_extension) || !in_array($file_extension, $allowed_extensions, true)) {
            // Attempt to detect via HTTP headers
            $head_response = wp_safe_remote_head($image_url, array('timeout' => 15));
            if (is_wp_error($head_response) || 200 !== wp_remote_retrieve_response_code($head_response)) {
                // Some servers block HEAD; try a lightweight GET request
                $head_response = wp_safe_remote_get($image_url, array(
                    'timeout' => 15,
                    'redirection' => 5,
                    'headers' => array('Range' => 'bytes=0-0')
                ));
            }

            if (!is_wp_error($head_response)) {
                $content_type_header = wp_remote_retrieve_header($head_response, 'content-type');
                if (!empty($content_type_header)) {
                    $mime_type = is_array($content_type_header) ? end($content_type_header) : $content_type_header;
                    $detected_ext = $this->guess_extension_from_mime($mime_type);
                    if (!empty($detected_ext)) {
                        $file_extension = $detected_ext;
                    }
                }
            }
        }

        // As a final fallback, rely on WordPress file type detection
        if (empty($file_extension) || !in_array($file_extension, $allowed_extensions, true)) {
            $filetype = wp_check_filetype_and_ext($temp_file, wp_basename($url_path));
            if (!empty($filetype['ext'])) {
                $file_extension = $filetype['ext'];
            }
            if (empty($mime_type) && !empty($filetype['type'])) {
                $mime_type = $filetype['type'];
            }
        }

        if (empty($file_extension)) {
            $file_extension = 'jpg';
        }

        if (empty($mime_type)) {
            $mime_type = $this->guess_mime_from_extension($file_extension);
        }

        // Build a safe filename from title (handles non-Latin titles)
        $base_name = sanitize_file_name($title);
        if (empty($base_name)) {
            $base_name = 'mosallas-featured-image';
        }
        $file_name = $base_name . '-' . time() . '.' . $file_extension;
        
        // Prepare file array
        $file = array(
            'name' => $file_name,
            'tmp_name' => $temp_file,
            'type' => $mime_type
        );
        
        // Upload the file
        $attachment_id = media_handle_sideload($file, $post_id, $title);
        
        // Clean up temp file
        if (is_wp_error($attachment_id)) {
            @unlink($temp_file);
            error_log('Mosallas: Failed to upload image - ' . $attachment_id->get_error_message());
            Mosallas_Logger::warning('⚠️ بارگذاری تصویر شاخص ناموفق بود', array(
                'error' => $attachment_id->get_error_message(),
                'source' => $image_url
            ));
            return 0;
        }
        
        // Set alt text
        update_post_meta($attachment_id, '_wp_attachment_image_alt', $title);
        
        // Generate attachment metadata (sizes) to ensure proper display
        $file_path = get_attached_file($attachment_id);
        if ($file_path) {
            $metadata = wp_generate_attachment_metadata($attachment_id, $file_path);
            if (!is_wp_error($metadata) && !empty($metadata)) {
                wp_update_attachment_metadata($attachment_id, $metadata);
            } else {
                error_log('Mosallas: Failed to generate attachment metadata for image ID ' . $attachment_id);
            }
        }
        
        return $attachment_id;
    }

    /**
     * Normalize remote image URLs to avoid download issues.
     */
    private function normalize_remote_image_url($url) {
        if (!is_string($url)) {
            return '';
        }

        $url = html_entity_decode(trim($url));
        if ($url === '') {
            return '';
        }

        // Replace spaces and control characters
        $url = preg_replace('/[\s]+/u', '%20', $url);

        $parsed = wp_parse_url($url);
        if (empty($parsed) || empty($parsed['host'])) {
            return $url;
        }

        $host = strtolower($parsed['host']);
        $local_hosts = array('localhost', '127.0.0.1', 'host.docker.internal');

        if (in_array($host, $local_hosts, true)) {
            $replacement_host = apply_filters('mosallas_minio_public_host', 's3api.mosallas.ir', $host, $parsed);
            if (!empty($replacement_host) && is_string($replacement_host)) {
                $parsed['scheme'] = 'https';
                $parsed['host'] = $replacement_host;
                unset($parsed['port']);
                $url = $this->build_url_from_parts($parsed);
            }
        }

        return $url;
    }

    /**
     * Build URL from parse_url parts
     */
    private function build_url_from_parts($parts) {
        $scheme = isset($parts['scheme']) ? $parts['scheme'] . '://' : '';
        $user = isset($parts['user']) ? $parts['user'] : '';
        $pass = isset($parts['pass']) ? ':' . $parts['pass'] : '';
        $auth = $user !== '' ? $user . $pass . '@' : '';
        $host = isset($parts['host']) ? $parts['host'] : '';
        $port = isset($parts['port']) ? ':' . $parts['port'] : '';
        $path = isset($parts['path']) ? $parts['path'] : '';
        $query = isset($parts['query']) ? '?' . $parts['query'] : '';
        $fragment = isset($parts['fragment']) ? '#' . $parts['fragment'] : '';

        return $scheme . $auth . $host . $port . $path . $query . $fragment;
    }

    /**
     * Map MIME type to file extension
     */
    private function guess_extension_from_mime($mime_type) {
        if (empty($mime_type)) {
            return '';
        }

        $mime_type = strtolower($mime_type);
        $map = array(
            'image/jpeg' => 'jpg',
            'image/jpg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
            'image/bmp' => 'bmp',
            'image/tiff' => 'tiff',
            'image/svg+xml' => 'svg',
            'image/avif' => 'avif',
            'image/heic' => 'heic',
            'image/heif' => 'heif'
        );

        return isset($map[$mime_type]) ? $map[$mime_type] : '';
    }

    /**
     * Map file extension to MIME type
     */
    private function guess_mime_from_extension($extension) {
        if (empty($extension)) {
            return 'image/jpeg';
        }

        $extension = strtolower($extension);
        $map = array(
            'jpg' => 'image/jpeg',
            'jpeg' => 'image/jpeg',
            'png' => 'image/png',
            'gif' => 'image/gif',
            'webp' => 'image/webp',
            'bmp' => 'image/bmp',
            'tiff' => 'image/tiff',
            'svg' => 'image/svg+xml',
            'avif' => 'image/avif',
            'heic' => 'image/heic',
            'heif' => 'image/heif'
        );

        return isset($map[$extension]) ? $map[$extension] : 'image/jpeg';
    }

    /**
     * Normalize FAQ schema input (stringified JSON or array/object) to an array.
     */
    private function normalize_faq_schema($schema) {
        if (empty($schema)) {
            return null;
        }

        if (is_string($schema)) {
            $decoded = json_decode($schema, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                return $decoded;
            }
            return null;
        }

        if (is_array($schema)) {
            return $schema;
        }

        if (is_object($schema)) {
            return json_decode(json_encode($schema), true);
        }

        return null;
    }

    /**
     * Extract and remove embedded JSON-LD from FAQ HTML, keeping schema separately.
     */
    private function strip_and_capture_faq_schema($faqContent, &$schemaTarget) {
        if (!is_string($faqContent)) {
            return $faqContent;
        }

        $pattern = '#<script[^>]*type=["\\\']application/ld\+json["\\\'][^>]*>(.*?)</script>#is';
        if (preg_match($pattern, $faqContent, $matches)) {
            $candidate = $this->normalize_faq_schema(html_entity_decode($matches[1]));
            if ($candidate) {
                $schemaTarget = $candidate;
            }

            $faqContent = preg_replace($pattern, '', $faqContent);
        }

        return $faqContent;
    }

    /**
     * Persist FAQ schema JSON-LD to post meta for head injection.
     */
    private function persist_faq_schema($post_id, $schema) {
        $normalized = $this->normalize_faq_schema($schema);

        if (empty($normalized)) {
            delete_post_meta($post_id, '_mosallas_faq_schema');
            return;
        }

        $encoded = wp_json_encode($normalized);
        if (!empty($encoded)) {
            update_post_meta($post_id, '_mosallas_faq_schema', $encoded);
        }
    }
    
    /**
     * Update SEO meta tags for Yoast SEO and Rank Math
     */
    private function update_seo_meta($post_id, $data) {
        $has_meta_title = array_key_exists('metaTitle', $data);
        $has_meta_description = array_key_exists('metaDescription', $data);
        $has_focus_keyword = array_key_exists('focusKeyword', $data);

        // Always persist to Mosallas custom meta (fallback storage for sites without SEO plugins)
        if ($has_meta_title) {
            $value = is_string($data['metaTitle']) ? trim($data['metaTitle']) : '';
            if ($value !== '') {
                update_post_meta($post_id, '_mosallas_meta_title', sanitize_text_field($value));
            }
        }
        if ($has_meta_description) {
            $value = is_string($data['metaDescription']) ? trim($data['metaDescription']) : '';
            if ($value !== '') {
                update_post_meta($post_id, '_mosallas_meta_description', sanitize_textarea_field($value));
            }
        }
        if ($has_focus_keyword) {
            $value = is_string($data['focusKeyword']) ? trim($data['focusKeyword']) : '';
            if ($value !== '') {
                update_post_meta($post_id, '_mosallas_focus_keyword', sanitize_text_field($value));
            }
        }

        // Update Yoast SEO meta
        if (class_exists('WPSEO_Meta')) {
            if ($has_meta_title) {
                $value = is_string($data['metaTitle']) ? trim($data['metaTitle']) : '';
                if ($value === '') {
                    delete_post_meta($post_id, '_yoast_wpseo_title');
                } else {
                    update_post_meta($post_id, '_yoast_wpseo_title', sanitize_text_field($value));
                }
            }

            if ($has_meta_description) {
                $value = is_string($data['metaDescription']) ? trim($data['metaDescription']) : '';
                if ($value === '') {
                    delete_post_meta($post_id, '_yoast_wpseo_metadesc');
                } else {
                    update_post_meta($post_id, '_yoast_wpseo_metadesc', sanitize_textarea_field($value));
                }
            }

            if ($has_focus_keyword) {
                $value = is_string($data['focusKeyword']) ? trim($data['focusKeyword']) : '';
                if ($value === '') {
                    delete_post_meta($post_id, '_yoast_wpseo_focuskw');
                } else {
                    update_post_meta($post_id, '_yoast_wpseo_focuskw', sanitize_text_field($value));
                }
            }
        }

        // Update Rank Math meta
        if (class_exists('RankMath')) {
            if ($has_meta_title) {
                $value = is_string($data['metaTitle']) ? trim($data['metaTitle']) : '';
                if ($value === '') {
                    delete_post_meta($post_id, 'rank_math_title');
                } else {
                    update_post_meta($post_id, 'rank_math_title', sanitize_text_field($value));
                }
            }

            if ($has_meta_description) {
                $value = is_string($data['metaDescription']) ? trim($data['metaDescription']) : '';
                if ($value === '') {
                    delete_post_meta($post_id, 'rank_math_description');
                } else {
                    update_post_meta($post_id, 'rank_math_description', sanitize_textarea_field($value));
                }
            }

            if ($has_focus_keyword) {
                $value = is_string($data['focusKeyword']) ? trim($data['focusKeyword']) : '';
                if ($value === '') {
                    delete_post_meta($post_id, 'rank_math_focus_keyword');
                } else {
                    update_post_meta($post_id, 'rank_math_focus_keyword', sanitize_text_field($value));
                }
            }
        }
    }
    
    /**
     * Get or create category using AI selection from Server
     */
    private function get_or_create_category($topic) {
        try {
            error_log('Mosallas Publisher: Starting AI category selection for topic: ' . $topic);

            // 1. Get all available categories from WordPress
            $categories = get_categories(array(
                'hide_empty' => false,
                'orderby' => 'name',
                'order' => 'ASC'
            ));

            $categories_payload = array();
            if (!empty($categories)) {
                foreach ($categories as $cat) {
                    $categories_payload[] = array(
                        'id' => $cat->term_id,
                        'name' => $cat->name,
                        'description' => $cat->description
                    );
                }
            } else {
                 error_log('Mosallas Publisher: No existing categories found in WordPress.');
            }

            // 2. Prepare article data
            $article_data = array(
                'title' => $this->current_article_title,
                'content' => $this->current_article_content,
                'topic' => $topic
            );

            // 3. Call Server API (Unified Select/Generate)
            $selection = $this->select_category_unified($categories_payload, $article_data);
            
            if (!$selection) {
                throw new Exception('AI Selection returned empty result');
            }

            $cat_id = isset($selection['categoryId']) ? intval($selection['categoryId']) : 0;
            $cat_name = isset($selection['categoryName']) ? trim($selection['categoryName']) : '';
            $reasoning = isset($selection['reasoning']) ? $selection['reasoning'] : '';

            // 4. Handle Selection
            if ($cat_id > 0) {
                // Scenario A: AI selected an existing category
                error_log("Mosallas Publisher: AI selected existing category ID: $cat_id ($cat_name)");
                Mosallas_Logger::info('🤖 هوش مصنوعی دسته‌بندی موجود را انتخاب کرد', array(
                    'category_id' => $cat_id,
                    'category_name' => $cat_name,
                    'reasoning' => $reasoning
                ));
                return array($cat_id);
            }

            // Scenario B: AI suggested a NEW category (categoryId == 0)
            if (!empty($cat_name)) {
                error_log("Mosallas Publisher: AI suggested NEW category: $cat_name");

                // Validate category name: must be 1-2 words, max 30 characters
                $cat_name = trim($cat_name);
                $word_count = count(preg_split('/\s+/', $cat_name, -1, PREG_SPLIT_NO_EMPTY));
                if ($word_count > 2 || mb_strlen($cat_name) > 30) {
                    error_log("Mosallas Publisher: Category name too long ($word_count words, " . mb_strlen($cat_name) . " chars), truncating");
                    // Take only first 2 words
                    $words = preg_split('/\s+/', $cat_name, -1, PREG_SPLIT_NO_EMPTY);
                    $cat_name = implode(' ', array_slice($words, 0, 2));
                    error_log("Mosallas Publisher: Truncated category name: $cat_name");
                }

                // Double check if it exists (race condition or manual create?)
                $existing_term = get_term_by('name', $cat_name, 'category');
                if ($existing_term) {
                    error_log("Mosallas Publisher: Category '$cat_name' actually exists with ID " . $existing_term->term_id . ". Using it.");
                    return array($existing_term->term_id);
                }

                // Also check by slug to prevent duplicates
                $potential_slug = sanitize_title($cat_name);
                $existing_by_slug = get_term_by('slug', $potential_slug, 'category');
                if ($existing_by_slug) {
                    error_log("Mosallas Publisher: Category with slug '$potential_slug' already exists with ID " . $existing_by_slug->term_id . ". Using it.");
                    return array($existing_by_slug->term_id);
                }

                // Create it
                if (!function_exists('wp_insert_term')) {
                    require_once ABSPATH . 'wp-admin/includes/taxonomy.php';
                }
                
                $insert_result = wp_insert_term($cat_name, 'category');
                
                if (is_wp_error($insert_result)) {
                    // Check if error is because term exists (edge case)
                    if ($insert_result->get_error_code() === 'term_exists') {
                        $existing_id = $insert_result->get_error_data('term_exists');
                        error_log("Mosallas Publisher: Category already existed, using ID: $existing_id");
                        return array($existing_id);
                    }
                    throw new Exception('Failed to create new category: ' . $insert_result->get_error_message());
                }
                
                $new_cat_id = $insert_result['term_id'];
                error_log("Mosallas Publisher: Created new category '$cat_name' with ID $new_cat_id");
                
                Mosallas_Logger::info('✨ دسته‌بندی جدید توسط هوش مصنوعی ساخته شد', array(
                    'category_id' => $new_cat_id,
                    'category_name' => $cat_name,
                    'reasoning' => $reasoning
                ));
                
                return array($new_cat_id);
            }

            throw new Exception("AI returned invalid selection (ID=0, Name empty)");

        } catch (Exception $e) {
            error_log('Mosallas Publisher: AI category selection failed - ' . $e->getMessage());
            Mosallas_Logger::warning('⚠️ خطا در انتخاب هوشمند دسته‌بندی (بازگشت به پیش‌فرض)', array(
                'error' => $e->getMessage()
            ));
            
            // CRITICAL: Strict fallback to default, NO RETRY, NO RANDOM
            return array(get_option('default_category', 1));
        }
    }

    /**
     * Unified Server API Call for Category Selection
     */
    private function select_category_unified($categories, $article_data) {
        $api_key = get_option('mosallas_api_key');
        $api_base = 'https://mosallas.ir/api/wordpress';
        
        if (empty($api_key)) {
            throw new Exception('API key not configured');
        }
        
        $request_data = array(
            'action' => 'select_category',
            'data' => array(
                'categories' => $categories,
                'articleTitle' => $article_data['title'],
                'articleContent' => $article_data['content'],
                'articleTopic' => $article_data['topic'],
                'articleId' => $this->current_article_id
            )
        );
        
        error_log('Mosallas Publisher: Calling select_category API unified with ' . count($categories) . ' categories');
        
        $response = wp_remote_post($api_base . '/webhook', array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode($request_data),
            'timeout' => 45 // Increased timeout for chained AI calls
        ));
        
        if (is_wp_error($response)) {
            throw new Exception('API request failed: ' . $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!$body || !isset($body['success']) || !$body['success']) {
            throw new Exception('API returned error: ' . json_encode($body));
        }
        
        $category = $body['category'] ?? null;
        return $category;
    }
    

    
    /**
     * Add internal links to related posts
     */
    private function add_internal_links($post_id, $existing_posts) {
        if (!is_array($existing_posts) || empty($existing_posts)) {
            return;
        }
        
        $post = get_post($post_id);
        if (!$post) {
            return;
        }
        
        $content = $post->post_content;
        $links_added = 0;
        $max_links = 3;
        
        foreach ($existing_posts as $existing_post) {
            if ($links_added >= $max_links) {
                break;
            }
            
            $keyword = $existing_post['focusKeyword'] ?? '';
            $post_id_link = $existing_post['wpPostId'] ?? 0;
            
            if (empty($keyword) || empty($post_id_link)) {
                continue;
            }
            
            // Check if keyword exists in content (case-insensitive)
            if (stripos($content, $keyword) !== false) {
                $permalink = get_permalink($post_id_link);
                
                // Create the link
                $link = '<a href="' . esc_url($permalink) . '" title="' . esc_attr($keyword) . '">' . 
                        esc_html($keyword) . '</a>';
                
                // Replace first occurrence of the keyword (case-insensitive)
                $content = preg_replace(
                    '/\b(' . preg_quote($keyword, '/') . ')\b/ui',
                    $link,
                    $content,
                    1,
                    $count
                );
                
                if ($count > 0) {
                    $links_added++;
                }
            }
        }
        
        // Update post if links were added
        if ($links_added > 0) {
            wp_update_post(array(
                'ID' => $post_id,
                'post_content' => $content
            ));
        }
    }
    
    /**
     * Get author ID for posts (defaults to first admin user)
     */
    private function get_author_id() {
        $current_user = get_current_user_id();
        
        if ($current_user > 0) {
            return $current_user;
        }
        
        // Get first admin user
        $admin_users = get_users(array(
            'role' => 'administrator',
            'number' => 1
        ));
        
        if (!empty($admin_users)) {
            return $admin_users[0]->ID;
        }
        
        return 1; // Fallback
    }
}



