<?php
/**
 * Mosallas Traffic Tracker
 * Records visitor statistics for connected WordPress sites
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mosallas_Tracker {
    /** @var Mosallas_Database */
    private $database;

    /** @var string */
    private $cookie_name = 'mosallas_vid';

    /** @var int */
    private $cookie_ttl;

    /** @var string[] */
    private $bot_signatures = array(
        'bot',
        'spider',
        'crawl',
        'slurp',
        'facebookexternalhit',
        'pingdom',
        'bingpreview',
        'mediapartners-google',
        'google-structured-data-testing-tool'
    );

    public function __construct(Mosallas_Database $database) {
        $this->database = $database;
        $this->cookie_ttl = YEAR_IN_SECONDS;
        add_action('template_redirect', array($this, 'maybe_track_visit'), 5);
    }

    /**
     * Track the current request if it matches our criteria
     */
    public function maybe_track_visit() {
        if (!$this->should_track_request()) {
            return;
        }

        $site_id = get_option('mosallas_site_id');
        if (empty($site_id)) {
            return;
        }

        $visitor_id = $this->get_or_set_visitor_id();
        if (empty($visitor_id)) {
            return;
        }

        $visitor_hash = hash('sha256', $site_id . '|' . $visitor_id);
        $is_generated = $this->is_mosallas_generated_post();
        $project_id = get_option('mosallas_project_id');

        // Prefer direct DB write if configured, otherwise fall back to webhook.
        if ($this->database && $this->database->is_connected()) {
            $this->database->record_site_visit(array(
                'site_id' => $site_id,
                'project_id' => !empty($project_id) ? $project_id : null,
                'stat_date' => gmdate('Y-m-d 00:00:00'),
                'visitor_hash' => $visitor_hash,
                'is_generated' => $is_generated,
            ));
            return;
        }

        $this->send_visit_to_webhook(array(
            'siteId' => $site_id,
            'visitorHash' => $visitor_hash,
            'visitedGenerated' => $is_generated,
            'statDate' => gmdate('Y-m-d'),
        ));
    }

    /**
     * Send a lightweight visit event to Mosallas webhook (no direct DB needed).
     */
    private function send_visit_to_webhook($data) {
        $api_key = get_option('mosallas_api_key');
        if (empty($api_key)) {
            return;
        }

        $webhook_url = get_option('mosallas_webhook_url', 'https://mosallas.ir/api/wordpress/webhook');
        if (empty($webhook_url)) {
            $webhook_url = 'https://mosallas.ir/api/wordpress/webhook';
        }

        $payload = array(
            'action' => 'track_visit',
            'data' => $data,
        );

        // Fire-and-forget (do not block page render).
        wp_remote_post($webhook_url, array(
            'headers' => array(
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode($payload),
            'timeout' => 2,
            'blocking' => false,
        ));
    }

    /**
     * Determine whether the current request should be tracked
     */
    private function should_track_request() {
        if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
            return false;
        }

        if (defined('REST_REQUEST') && REST_REQUEST) {
            return false;
        }

        if (PHP_SAPI === 'cli') {
            return false;
        }

        $method = isset($_SERVER['REQUEST_METHOD']) ? strtoupper($_SERVER['REQUEST_METHOD']) : 'GET';
        if (!in_array($method, array('GET', 'HEAD'), true)) {
            return false;
        }

        if (is_404() || is_search() || is_feed() || is_trackback() || is_preview()) {
            return false;
        }

        if ($this->is_bot_request()) {
            return false;
        }

        $api_key = get_option('mosallas_api_key');
        $connection_status = get_option('mosallas_connection_status');

        if (empty($api_key) || $connection_status !== 'connected') {
            return false;
        }

        return true;
    }

    /**
     * Generate or reuse a lightweight visitor identifier stored in a cookie
     */
    private function get_or_set_visitor_id() {
        $existing = isset($_COOKIE[$this->cookie_name]) ? sanitize_text_field(wp_unslash($_COOKIE[$this->cookie_name])) : '';
        if (!empty($existing)) {
            return $existing;
        }

        if (headers_sent()) {
            return '';
        }

        $new_id = wp_generate_uuid4();
        $cookie_args = array(
            'expires' => time() + $this->cookie_ttl,
            'path' => defined('COOKIEPATH') && COOKIEPATH ? COOKIEPATH : '/',
            'domain' => defined('COOKIE_DOMAIN') && COOKIE_DOMAIN ? COOKIE_DOMAIN : '',
            'secure' => is_ssl(),
            'httponly' => false,
            'samesite' => 'Lax',
        );

        setcookie($this->cookie_name, $new_id, $cookie_args);
        $_COOKIE[$this->cookie_name] = $new_id;

        return $new_id;
    }

    /**
     * Check if current user agent is likely a bot
     */
    private function is_bot_request() {
        if (empty($_SERVER['HTTP_USER_AGENT'])) {
            return false;
        }

        $agent = strtolower(sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])));

        foreach ($this->bot_signatures as $signature) {
            if (false !== strpos($agent, $signature)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Determine whether the current singular post was generated by Mosallas
     */
    private function is_mosallas_generated_post() {
        if (!is_singular('post')) {
            return false;
        }

        $post_id = get_queried_object_id();
        if (empty($post_id)) {
            return false;
        }

        return (bool) get_post_meta($post_id, '_mosallas_generated', true);
    }
}
