<?php
/**
 * Mosallas Plugin Updater
 * Provides automatic update checks against the Mosallas distribution server.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Mosallas_Updater {
    /**
     * Singleton instance.
     *
     * @var Mosallas_Updater|null
     */
    private static $instance = null;

    /**
     * Remote JSON manifest URL.
     *
     * @var string
     */
    private $update_url;

    /**
     * Plugin basename (slug/file path)
     *
     * @var string
     */
    private $plugin_file;

    /**
     * Plugin slug.
     *
     * @var string
     */
    private $slug;

    /**
     * Transient cache key for remote response.
     *
     * @var string
     */
    private $cache_key;

    /**
     * Cache lifetime.
     *
     * @var int
     */
    private $cache_ttl;

    /**
     * Flag to ensure hooks are registered once.
     *
     * @var bool
     */
    private $initialized = false;

    /**
     * Retrieve singleton instance.
     *
     * @return Mosallas_Updater
     */
    public static function instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Mosallas_Updater constructor.
     */
    private function __construct() {
        $default_url = 'https://mosallas.ir/plugin-update.json';

        $this->update_url = apply_filters('mosallas_plugin_update_json_url', $default_url);
        $this->plugin_file = MOSALLAS_PLUGIN_BASENAME;
        $this->slug = dirname($this->plugin_file);
        $this->cache_key = 'mosallas_plugin_update_' . md5($this->update_url);

        $default_ttl = (defined('MINUTE_IN_SECONDS') ? 15 * constant('MINUTE_IN_SECONDS') : 900); // 15 minutes
        $this->cache_ttl = apply_filters('mosallas_plugin_update_cache_ttl', $default_ttl);
    }

    /**
     * Register WordPress hooks for update handling.
     */
    public function init() {
        if ($this->initialized) {
            return;
        }

    add_filter('pre_set_site_transient_update_plugins', array($this, 'check_for_update'));
    add_filter('plugins_api', array($this, 'plugins_api_handler'), 10, 3);
    add_action('upgrader_process_complete', array($this, 'clear_update_cache'), 10, 2);
    add_filter('auto_update_plugin', array($this, 'enable_auto_updates'), 10, 2);

        $this->initialized = true;
    }

    /**
     * Force-enable WordPress automatic updates for this plugin.
     *
     * @param bool   $update
     * @param object $item
     * @return bool
     */
    public function enable_auto_updates($update, $item) {
        if (isset($item->plugin) && $item->plugin === $this->plugin_file) {
            return true;
        }

        return $update;
    }

    /**
     * Inject update information into WordPress update transient.
     *
     * @param stdClass $transient
     * @return stdClass
     */
    public function check_for_update($transient) {
        if (!is_object($transient)) {
            $transient = new stdClass();
        }

        if (empty($transient->checked) || !isset($transient->checked[$this->plugin_file])) {
            return $transient;
        }

        $remote = $this->get_update_data();
        if (!$remote) {
            return $transient;
        }

        $remote_version = $this->extract_remote_version($remote);
        if (!$remote_version || version_compare($remote_version, MOSALLAS_VERSION, '<=')) {
            return $transient;
        }

        $package = $this->extract_download_url($remote);
        if (empty($package)) {
            return $transient;
        }

        $plugin_info = (object) array(
            'slug'         => $this->slug,
            'plugin'       => $this->plugin_file,
            'new_version'  => $remote_version,
            'package'      => $package,
            'url'          => $this->extract_homepage($remote),
            'tested'       => $remote['tested'] ?? '',
            'requires'     => $remote['requires'] ?? '',
            'requires_php' => $remote['requires_php'] ?? '',
            'icons'        => $remote['icons'] ?? array(),
            'banners'      => $remote['banners'] ?? array(),
            'banners_rtl'  => $remote['banners_rtl'] ?? array(),
        );

        $transient->response[$this->plugin_file] = $plugin_info;
        $transient->last_checked = time();
        $transient->checked[$this->plugin_file] = MOSALLAS_VERSION;

        return $transient;
    }

    /**
     * Provide detailed plugin information within the WordPress updater UI.
     *
     * @param false|object $result
     * @param string $action
     * @param object $args
     * @return false|object
     */
    public function plugins_api_handler($result, $action, $args) {
        if ($action !== 'plugin_information') {
            return $result;
        }

        if (!isset($args->slug) || $args->slug !== $this->slug) {
            return $result;
        }

        $remote = $this->get_update_data(true);
        if (!$remote) {
            return $result;
        }

        $info = new stdClass();
        $info->name = $remote['name'] ?? 'Mosallas.ir - SEO Content Automation';
        $info->slug = $this->slug;
        $info->version = $this->extract_remote_version($remote) ?: MOSALLAS_VERSION;
        $info->author = $remote['author'] ?? '<a href="https://mosallas.ir">Mosallas.ir</a>';
        $info->author_profile = $remote['author_profile'] ?? 'https://mosallas.ir';
        $info->requires = $remote['requires'] ?? '6.0';
        $info->tested = $remote['tested'] ?? '';
        $info->requires_php = $remote['requires_php'] ?? '7.4';
        $info->homepage = $this->extract_homepage($remote);
        $info->download_link = $this->extract_download_url($remote);
        $info->package = $info->download_link;
        $info->sections = $remote['sections'] ?? array(
            'description' => $remote['description'] ?? __('افزونه Mosallas برای خودکارسازی تولید و انتشار محتوا.', 'mosallas'),
        );
        $info->banners = $remote['banners'] ?? array();
        $info->banners_rtl = $remote['banners_rtl'] ?? array();
        $info->icons = $remote['icons'] ?? array();
        $info->last_updated = $remote['last_updated'] ?? '';
        $info->rating = $remote['rating'] ?? 0;
        $info->num_ratings = $remote['num_ratings'] ?? 0;
        $info->active_installs = $remote['active_installs'] ?? 0;

        return $info;
    }

    /**
     * Clear cached update data after installation completes.
     *
     * @param WP_Upgrader $upgrader
     * @param array $options
     * @return void
     */
    public function clear_update_cache($upgrader, $options) {
        if (!isset($options['type']) || $options['type'] !== 'plugin') {
            return;
        }

        if (isset($options['plugins']) && is_array($options['plugins']) && in_array($this->plugin_file, $options['plugins'], true)) {
            delete_site_transient($this->cache_key);
        }
    }

    /**
     * Fetch update data from the Mosallas API.
     *
     * @param bool $force
     * @return array|null
     */
    private function get_update_data($force = false) {
        if (!$force) {
            $cached = get_site_transient($this->cache_key);
            if ($cached !== false) {
                return $cached;
            }
        }

        $response = wp_remote_get($this->update_url, array(
            'timeout' => 15,
            'headers' => array(
                'Accept' => 'application/json',
            ),
        ));

        if (is_wp_error($response)) {
            Mosallas_Logger::error('❌ خطا در بررسی به‌روزرسانی افزونه Mosallas', array(
                'error' => $response->get_error_message(),
            ));
            return null;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($status_code !== 200 || empty($body)) {
            Mosallas_Logger::warning('⚠️ پاسخ نامعتبر از سرور به‌روزرسانی Mosallas', array(
                'status_code' => $status_code,
                'body'        => $body,
            ));
            return null;
        }

        $decoded = json_decode($body, true);
        if (!is_array($decoded)) {
            Mosallas_Logger::warning('⚠️ محتوای JSON نامعتبر برای به‌روزرسانی Mosallas', array(
                'body' => $body,
            ));
            return null;
        }

        set_site_transient($this->cache_key, $decoded, $this->cache_ttl);

        return $decoded;
    }

    /**
     * Extract version value from remote payload.
     *
     * @param array $remote
     * @return string|null
     */
    private function extract_remote_version($remote) {
        if (isset($remote['version'])) {
            return $remote['version'];
        }

        if (isset($remote['new_version'])) {
            return $remote['new_version'];
        }

        if (isset($remote['data']) && is_array($remote['data'])) {
            if (isset($remote['data']['version'])) {
                return $remote['data']['version'];
            }
            if (isset($remote['data']['new_version'])) {
                return $remote['data']['new_version'];
            }
        }

        return null;
    }

    /**
     * Extract download package URL from remote payload.
     *
     * @param array $remote
     * @return string|null
     */
    private function extract_download_url($remote) {
        if (!empty($remote['download_url'])) {
            return esc_url_raw($remote['download_url']);
        }

        if (!empty($remote['package'])) {
            return esc_url_raw($remote['package']);
        }

        if (isset($remote['data']) && is_array($remote['data'])) {
            if (!empty($remote['data']['download_url'])) {
                return esc_url_raw($remote['data']['download_url']);
            }
            if (!empty($remote['data']['package'])) {
                return esc_url_raw($remote['data']['package']);
            }
        }

        return null;
    }

    /**
     * Determine homepage/support URL.
     *
     * @param array $remote
     * @return string
     */
    private function extract_homepage($remote) {
        if (!empty($remote['homepage'])) {
            return esc_url_raw($remote['homepage']);
        }

        if (!empty($remote['url'])) {
            return esc_url_raw($remote['url']);
        }

        if (isset($remote['data']) && is_array($remote['data'])) {
            if (!empty($remote['data']['homepage'])) {
                return esc_url_raw($remote['data']['homepage']);
            }
            if (!empty($remote['data']['url'])) {
                return esc_url_raw($remote['data']['url']);
            }
        }

        return 'https://mosallas.ir';
    }
}
