<?php
/**
 * Plugin Name: Mosallas.ir - SEO Content Automation
 * Plugin URI: https://mosallas.ir
 * Description: Automatically publish AI-generated SEO articles from Mosallas.ir
 * Version: 1.4.0
 * Author: Mosallas.ir
 * Author URI: https://mosallas.ir
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: mosallas
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('MOSALLAS_VERSION', '1.4.0');
define('MOSALLAS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('MOSALLAS_PLUGIN_URL', plugin_dir_url(__FILE__));
define('MOSALLAS_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Include required files
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-logger.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-llms-txt.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-api.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-admin.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-publisher.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-database.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-tracker.php';
require_once MOSALLAS_PLUGIN_DIR . 'includes/class-mosallas-updater.php';

/**
 * Initialize plugin
 */
function mosallas_init() {
    // Initialize classes
    $admin = new Mosallas_Admin();
    $api = new Mosallas_API();
    $database = new Mosallas_Database();
    new Mosallas_Tracker($database);
    $llms_txt = new Mosallas_LLMs_Txt();
    $llms_txt->init();

    // Register plugin updater
    Mosallas_Updater::instance()->init();
    
    // Register admin menu
    add_action('admin_menu', array($admin, 'add_admin_menu'));
    
    // Register admin scripts and styles
    add_action('admin_enqueue_scripts', array($admin, 'enqueue_admin_assets'));
    
    // Register REST API endpoints
    add_action('rest_api_init', array($api, 'register_routes'));
    
    // Register FAQ Schema field in REST API
    add_action('rest_api_init', 'mosallas_register_faq_schema_field');
    
    // Register AJAX handlers for logs
    add_action('wp_ajax_mosallas_get_logs', array($admin, 'ajax_get_logs'));
    add_action('wp_ajax_mosallas_clear_logs', array($admin, 'ajax_clear_logs'));
    add_action('wp_ajax_mosallas_download_logs', array($admin, 'ajax_download_logs'));
    add_action('wp_ajax_mosallas_update_connection_status', array($admin, 'ajax_update_connection_status'));
    add_action('wp_ajax_mosallas_confirm_connection', array($admin, 'ajax_confirm_connection'));
    
    // Register AJAX handlers for database sync
    add_action('wp_ajax_mosallas_sync_articles', array($database, 'sync_articles'));
    add_action('wp_ajax_mosallas_sync_products', array($database, 'sync_products'));
    
    // Schedule automatic sync
    add_action('mosallas_auto_sync', array($database, 'sync_articles'));
    add_action('mosallas_auto_sync', array($database, 'sync_products'));
    
    // Hook into post save to sync individual posts
    add_action('save_post', function($post_id) use ($database) {
        if (wp_is_post_revision($post_id) || wp_is_post_autosave($post_id)) {
            return;
        }
        
        $post = get_post($post_id);
        if ($post && in_array($post->post_status, array('publish', 'private'))) {
            $database->save_article_to_db($post);
        }
    });
    
    // Hook into product save for WooCommerce
    add_action('woocommerce_update_product', function($product_id) use ($database) {
        $product = wc_get_product($product_id);
        if ($product) {
            $database->save_product_to_db($product);
        }
    });

    // Output stored FAQ schema in the document head
    add_action('wp_head', 'mosallas_output_faq_schema', 20);

    // Inject meta description for sites without Yoast/RankMath
    add_action('wp_head', 'mosallas_inject_seo_meta', 1);
}
add_action('plugins_loaded', 'mosallas_init');

/**
 * Register custom REST API fields
 */
function mosallas_register_faq_schema_field() {
    register_rest_field('post', 'faq_schema', array(
        'get_callback' => function($post) {
            $schema = get_post_meta($post['id'], '_mosallas_faq_schema', true);
            if (empty($schema)) {
                return null;
            }
            // If stored as JSON string, decode it. If array, return as is.
            return is_string($schema) ? json_decode($schema, true) : $schema;
        },
        'schema' => array(
            'description' => 'Custom FAQ Schema JSON-LD',
            'type'        => 'object',
            'context'     => array('embeddable' => true),
        ),
    ));
}

/**
 * Activation hook
 */
function mosallas_activate() {
    // Flush rewrite rules
    flush_rewrite_rules();
    
    // Set default options
    if (get_option('mosallas_api_key') === false) {
        add_option('mosallas_api_key', '');
    }
    
    // Set default database options
    if (get_option('mosallas_db_host') === false) {
        add_option('mosallas_db_host', 'localhost');
    }
    if (get_option('mosallas_db_port') === false) {
        add_option('mosallas_db_port', '5432');
    }
    if (get_option('mosallas_db_name') === false) {
        add_option('mosallas_db_name', 'mosallas');
    }
    if (get_option('mosallas_db_user') === false) {
        add_option('mosallas_db_user', '');
    }
    if (get_option('mosallas_db_password') === false) {
        add_option('mosallas_db_password', '');
    }
    if (get_option('mosallas_site_id') === false) {
        add_option('mosallas_site_id', wp_generate_uuid4());
    }
    if (get_option('mosallas_connection_status') === false) {
        add_option('mosallas_connection_status', 'pending');
    }
    if (get_option('mosallas_connection_message') === false) {
        add_option('mosallas_connection_message', __('برای فعال‌سازی ارتباط، روی دکمه «اتصال به وبسایت» کلیک کنید.', 'mosallas'));
    }
    if (get_option('mosallas_connection_updated_at') === false) {
        add_option('mosallas_connection_updated_at', current_time('mysql'));
    }
    if (get_option('mosallas_project_id') === false) {
        add_option('mosallas_project_id', '');
    }

    if (get_option('mosallas_project_system_prompt') === false) {
        add_option('mosallas_project_system_prompt', '');
    }

    if (get_option('mosallas_llms_txt_cached') === false) {
        add_option('mosallas_llms_txt_cached', '', false);
    }
    if (get_option('mosallas_llms_txt_cached_at') === false) {
        add_option('mosallas_llms_txt_cached_at', 0, false);
    }
    
    // Confirm connection if API key exists
    $api_key = get_option('mosallas_api_key');
    if (!empty($api_key)) {
        $api = new Mosallas_API();
        $api->confirm_connection();
    }

    // Best-effort: refresh via webhook and write to site root on activation.
    if (class_exists('Mosallas_LLMs_Txt')) {
        $refresh = Mosallas_LLMs_Txt::refresh_llms_txt_via_webhook();
        if (!empty($refresh['success'])) {
            Mosallas_Logger::info('📄 llms.txt refreshed', $refresh);
        } else {
            Mosallas_Logger::warning('⚠️ llms.txt refresh failed', $refresh);
        }

        $write = Mosallas_LLMs_Txt::write_llms_txt_to_root();
        if (!empty($write['success'])) {
            Mosallas_Logger::info('📄 llms.txt created', $write);
        } else {
            Mosallas_Logger::warning('⚠️ llms.txt not written', $write);
        }
    }
    
    // Schedule automatic sync
    $database = new Mosallas_Database();
    $database->schedule_auto_sync();
}
register_activation_hook(__FILE__, 'mosallas_activate');

/**
 * Deactivation hook
 */
function mosallas_deactivate() {
    // Flush rewrite rules
    flush_rewrite_rules();
    
    // Unschedule automatic sync
    $database = new Mosallas_Database();
    $database->unschedule_auto_sync();
}
register_deactivation_hook(__FILE__, 'mosallas_deactivate');

/**
 * Uninstall hook
 */
function mosallas_uninstall() {
    // Remove options
    delete_option('mosallas_api_key');
    delete_option('mosallas_connection_status');
    delete_option('mosallas_connection_message');
    delete_option('mosallas_connection_updated_at');
    delete_option('mosallas_connection_error');
    delete_option('mosallas_db_host');
    delete_option('mosallas_db_port');
    delete_option('mosallas_db_name');
    delete_option('mosallas_db_user');
    delete_option('mosallas_db_password');
    delete_option('mosallas_site_id');
    delete_option('mosallas_project_id');
    delete_option('mosallas_project_system_prompt');
    delete_option('mosallas_llms_txt_cached');
    delete_option('mosallas_llms_txt_cached_at');
}
register_uninstall_hook(__FILE__, 'mosallas_uninstall');

/**
 * Render FAQ schema JSON-LD saved in post meta.
 */
function mosallas_output_faq_schema() {
    if (!is_singular()) {
        return;
    }

    $post_id = get_the_ID();
    if (!$post_id) {
        return;
    }

    $schema = get_post_meta($post_id, '_mosallas_faq_schema', true);
    if (empty($schema)) {
        return;
    }

    $decoded = is_string($schema) ? json_decode($schema, true) : $schema;
    if (empty($decoded)) {
        return;
    }

    $encoded = wp_json_encode($decoded);
    if (empty($encoded)) {
        return;
    }

    echo '<script type="application/ld+json">' . $encoded . '</script>' . "\n";
}

/**
 * Inject meta description and title for sites without Yoast/RankMath.
 * This provides SEO fallback for Mosallas-generated content.
 */
function mosallas_inject_seo_meta() {
    if (!is_singular('post')) {
        return;
    }

    // Skip if Yoast or RankMath is handling meta tags
    if (class_exists('WPSEO_Meta') || class_exists('RankMath')) {
        return;
    }

    $post_id = get_the_ID();
    if (!$post_id) {
        return;
    }

    // Output meta description from Mosallas storage
    $meta_desc = get_post_meta($post_id, '_mosallas_meta_description', true);
    if (!empty($meta_desc)) {
        echo '<meta name="description" content="' . esc_attr($meta_desc) . '" />' . "\n";
    }
}
?>
